<?php
/**
 * URFU_AJAX
 *
 * AJAX Event Handler
 *
 * @class    URFU_AJAX
 * @version  1.0.0
 * @package  UserRegistrationFileUpload/Classes
 * @category Class
 * @author   WPEverest
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * URFU_AJAX Class
 */
class URFU_AJAX {

	/**
	 * Hooks in ajax handlers
	 */
	public static function init() {

		self::add_ajax_events();

	}

	/**
	 * Hook in methods - uses WordPress ajax handlers (admin-ajax)
	 */
	public static function add_ajax_events() {

		$ajax_events = array(
			'method_upload' => true,
			'admin_upload'  => true,
		);

		foreach ( $ajax_events as $ajax_event => $nopriv ) {

			add_action( 'wp_ajax_user_registration_file_upload_' . $ajax_event, array( __CLASS__, $ajax_event ) );

			if ( $nopriv ) {

				add_action(
					'wp_ajax_nopriv_user_registration_file_upload_' . $ajax_event,
					array(
						__CLASS__,
						$ajax_event,
					)
				);
			}
		}
	}

	/**
	 * Handles the multiple files upload at once or one by one and push eact of them into an array
	 *
	 * @param array $files Uploaded files.
	 * @return array
	 */
	public static function handle_multiple_file( $files ) {
		// Handles multiple files in single upload request.
		$files1 = $files;
		$files2 = array();

		foreach ( $files1 as $input => $infoArr ) {
			$filesByInput = array();

			foreach ( $infoArr as $key => $valueArr ) {

				if ( is_array( $valueArr ) ) {
					// file input "multiple".

					foreach ( $valueArr as $i => $value ) {
						$filesByInput[ $i ][ $key ] = $value;
					}
				} else { // -> string, normal file input
					$filesByInput[] = $infoArr;
					break;
				}
			}
			$files2 = array_merge( $files2, $filesByInput );
		}

		$upload = array();

		foreach ( $files2 as $file ) {
			// let's filter empty & errors.

			if ( ! $file['error'] ) {
				$upload[] = $file;
			}
		}

		return $upload;
	}

	/**
	 * User input dropped function
	 *
	 * @var array $_REQUEST
	 * @var array $_FILES
	 */
	public static function method_upload() {

		check_ajax_referer( 'urfu_file_upload_nonce', 'security' );

		$nonce = isset( $_REQUEST['security'] ) ? sanitize_text_field( wp_unslash( $_REQUEST['security'] ) ) : '';

		$flag = wp_verify_nonce( $nonce, 'urfu_file_upload_nonce' );

		if ( true != $flag || is_wp_error( $flag ) ) {

			wp_send_json_error(
				array(
					'message' => __( 'Nonce error, please reload.', 'user-registration-file-upload' ),
				)
			);
		}

		$upload        = self::handle_multiple_file( $_FILES );
		$response_node = array();
		$upload_files  = array();

		for ( $i = 0; $i < sizeOf( $upload ); $i++ ) {

			if ( ! isset( $upload[ $i ]['size'] ) || ( isset( $upload[ $i ]['size'] ) && $upload[ $i ]['size'] < 1 ) ) {

				wp_send_json_error(
					array(
						'message' => __( 'Empty file.', 'user-registration-file-upload' ),
					)
				);
			}

			$form_id = 0;

			// In case of registration form, a form_id parameter will be sent to identify the form,
			if ( isset( $_REQUEST['form_id'] ) && 'undefined' !== $_REQUEST['form_id'] ) {
				$form_id = $_REQUEST['form_id'];
			} else {
				// In case of edit profile if form_id is not sent as a paramater then extract the form id from the current user_id.
				$user_id = get_current_user_id();

				if ( $user_id > 0 ) {
					$form_id = ur_get_form_id_by_userid( $user_id );
				}
			}

			$field_name           = isset( $_REQUEST['field_name'] ) ? str_replace( 'user_registration_', '', $_REQUEST['field_name'] ) : 0; //phpcs:ignore.
			$field_data = ur_get_field_data_by_field_name( $form_id, $field_name );

			$global_valid_file_size = get_option( 'user_registration_file_upload_setting_max_file_size', array() );

			if ( empty( $global_valid_file_size ) ) {
				$global_valid_file_size = wp_max_upload_size();
			}

			$valid_file_size = ! empty( $field_data['advance_setting']->max_upload_size ) ? $field_data['advance_setting']->max_upload_size : $global_valid_file_size;

			if ( $upload[ $i ]['size'] > $valid_file_size * 1024 ) {
				wp_send_json_error(
					array(
						'message' => sprintf( __( 'File size exceed, please upload file less than %d KB.', 'user-registration-file-upload' ), $valid_file_size ),
					)
				);
			}

			$ur_form = UR()->form->get_form( $form_id );

			if ( empty( $ur_form ) ) {
				wp_send_json_error(
					array(
						'message' => __( 'The form you are trying to submit not found.', 'user-registration-file-upload' ),
					)
				);
			}

			if ( empty( $field_data ) ) {
				wp_send_json_error(
					array(
						'message' => __( 'The file upload field you are trying to upload files to not found in the current form.', 'user-registration-file-upload' ),
					)
				);
			}

			// valid extension for file upload.
			$default_file_types = array_flip( urfu_get_valid_file_type() );

			$global_valid_extensions = get_option( 'user_registration_file_upload_setting_valid_file_type', array() );

			if ( empty( $global_valid_extensions ) ) {
				$global_valid_extensions = $default_file_types;
			}

			$valid_extensions = ! empty( $field_data['advance_setting']->valid_file_type ) ? $field_data['advance_setting']->valid_file_type : $global_valid_extensions;

			$file_extension = strtolower( pathinfo( $upload[ $i ]['name'], PATHINFO_EXTENSION ) );
			if ( function_exists( 'mime_content_type' ) ) {
				$file_mime_type = isset( $upload[ $i ]['tmp_name'] ) ? mime_content_type( $upload[ $i ]['tmp_name'] ) : '';
			} else {
				$upload_file_info = isset( $upload[ $i ]['tmp_name'] ) ? wp_check_filetype_and_ext( $upload[ $i ]['tmp_name'], $upload[ $i ]['name'] ) : '';
				$file_mime_type   = ! empty( $upload_file_info ) ? $upload_file_info['type'] : '';
			}

			// Validates if the uploaded file has the acceptable extension.
			$valid_ext = array();

			foreach ( $valid_extensions as $file_type ) {
				$key = array_search( $file_type, $default_file_types, true );

				if ( $key ) {
					array_push( $valid_ext, $key );
				}

				if ( 'image/jpeg' === $file_type ) {
					array_push( $valid_ext, 'jpg' );
				}
			}

			if ( ! in_array( $file_mime_type, array_values( $valid_extensions ), true ) ) {
				wp_send_json_error(
					array(
						'message' => __( 'Invalid file type, please contact with site administrator.', 'user-registration-file-upload' ),
					)
				);
			}

			if ( ! in_array( $file_extension, $valid_ext, true ) ) {
				wp_send_json_error(
					array(
						'message' => __( 'Invalid file type, please contact with site administrator.', 'user-registration-file-upload' ),
					)
				);
			}

			$upload_path = ur_get_tmp_dir();

			// Checks if the upload directory has the write premission.
			if ( ! wp_is_writable( $upload_path ) ) {
				wp_send_json_error(
					array(
						'message' => __( 'Upload path permission deny.', 'user-registration-file-upload' ),
					)
				);

			}
			$upload_path = $upload_path . '/';

			$file_name = wp_unique_filename( $upload_path, $upload[ $i ]['name'] );

			$file_path = $upload_path . sanitize_file_name( $file_name );

			if ( move_uploaded_file( $upload[ $i ]['tmp_name'], $file_path ) ) {

				$attachment_id      = wp_rand();
				$upload_files[ $i ] = $attachment_id;

				$files[ $i ] = array(
					'file_name'      => $file_name,
					'file_path'      => $file_path,
					'file_extension' => $file_extension,
				);

				$response_node[ $i ]  = '<a class="dz-remove urfu-remove-file" href="javascript:undefined;" data-dz-remove="" data-file-name="' . $file_name . '" data-attachment-id="' . $attachment_id . '" data-field-name="' . $field_name . '">Remove file</a>';
				$response_node[ $i ] .= '<input type="hidden" name="urfu_uploaded_file_' . $field_name . '[]" class="urfu-uploaded-file" value="' . crypt_the_string( maybe_serialize( $files[ $i ] ), 'e' ) . '">';

			} else {
				wp_send_json_error(
					array(
						'message' => __( 'File cannot be uploaded', 'user-registration-file-upload' ),
					)
				);
			}
		}
		ur_clean_tmp_files();

		wp_send_json_success(
			array(
				'message'      => $response_node,
				'upload_files' => $upload_files,
			)
		);
	}

	/**
	 * User input dropped function
	 *
	 * @var array $_REQUEST
	 * @var array $_FILES
	 */
	public static function admin_upload() {

		check_ajax_referer( 'urfu_admin_upload_nonce', 'security' );

		$nonce = isset( $_REQUEST['security'] ) ? sanitize_text_field( wp_unslash( $_REQUEST['security'] ) ) : '';

		$flag = wp_verify_nonce( $nonce, 'urfu_admin_upload_nonce' );

		if ( true != $flag || is_wp_error( $flag ) ) {

			wp_send_json_error(
				array(
					'message' => __( 'Nonce error, please reload.', 'user-registration-file-upload' ),
				)
			);
		}

		$upload_path = apply_filters( 'user_registration_file_upload_url', UR_UPLOAD_PATH . 'file-uploads' ); /*
		Get path of upload dir of WordPress*/
		// Checks if the upload directory exists and create one if not.
		if ( ! file_exists( $upload_path ) ) {
			wp_mkdir_p( $upload_path );
		}

		// Checks if the upload directory has the write premission.
		if ( ! wp_is_writable( $upload_path ) ) {
			wp_send_json_error(
				array(
					'message' => __( 'Upload path permission deny.', 'user-registration-file-upload' ),
				)
			);

		}
		$attachment = isset( $_POST ) ? $_POST : array();

			$upload_path = $upload_path . '/';

			// Check the type of file. We'll use this as the 'post_mime_type'.
			$file_ext = wp_check_filetype( basename( $attachment['filename'] ), null );

			$file_name = wp_unique_filename( $upload_path, $attachment['filename'] );

			$file_path = $upload_path . sanitize_file_name( $file_name );

			$attachment_path = get_attached_file( $attachment['id'] );

		if ( copy( $attachment_path, $file_path ) ) {

			$attachment_id = wp_insert_attachment(
				array(
					'guid'           => $file_path,
					'post_mime_type' => $file_ext['type'],
					'post_title'     => preg_replace( '/\.[^.]+$/', '', sanitize_file_name( $file_name ) ),
					'post_content'   => '',
					'post_status'    => 'inherit',
				),
				$file_path
			);
			if ( is_wp_error( $attachment_id ) ) {

				wp_send_json_error(
					array(

						'message' => $attachment_id->get_error_message(),
					)
				);
			}

			include_once ABSPATH . 'wp-admin/includes/image.php';

			wp_send_json_success(
				array(
					'attachment_id'  => $attachment_id,
					'attachment_url' => wp_get_attachment_url( $attachment_id ),
					'file_name'      => basename( get_attached_file( $attachment_id ) ),
				)
			);
		}
	}
}

URFU_AJAX::init();
