<?php
/**
 * UserRegistrationFileUpload Admin.
 *
 * @class    URFU_Admin
 * @version  1.0.0
 * @package  UserRegistrationFileUpload/Admin
 * @category Admin
 * @author   WPEverest
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * URFU_Admin Class
 */
class URFU_Admin {

	/**
	 * Hook in tabs.
	 */
	public function __construct() {
		urfu_check_plugin_compatibility();
		$message = urfu_is_compatible();
		if ( 'YES' !== $message ) {
			return;
		}
		add_filter( 'user_registration_get_settings_pages', array( $this, 'add_file_upload_setting' ), 10, 1 );
		add_filter( 'user_registration_registered_form_fields', 'urfu_registered_form_fields', 10, 1 );
		add_filter( 'user_registration_form_field_file_path', 'urfu_form_field_file', 10, 1 );
		add_filter(	'user_registration_file_admin_template', array( $this, 'user_registration_file_admin_template' ), 10, 1	);
		add_action( 'admin_enqueue_scripts', array( $this, 'admin_styles' ) );
		add_action( 'admin_enqueue_scripts', array( $this, 'admin_scripts' ) );
		add_action( 'user_registration_profile_field_file', array( $this, 'user_registration_profile_field' ) );
		add_action(	'user_registration_profile_field_filter_file', array( $this, 'user_registration_profile_field_filter' ) );
		add_filter( 'wp_handle_upload_prefilter', 'urfu_upload_file_type', 10, 1 );

		add_action( 'admin_init', array( $this, 'download_upload_data' ) );
		add_action( 'admin_init', array( $this, 'admin_redirects' ) );
		add_action( 'user_registration_after_admin_save_profile_validation', array( $this, 'user_registration_after_admin_save_profile_validation' ), 10, 2 );

		add_filter( 'user_registration_field_options_general_settings', array( $this, 'field_settings' ), 10, 2 );
		add_filter(	'user_registration_file_advance_class',	array( $this, 'user_registration_file_advance_class' ), 10,	1 );

		add_action( 'user_registration_single_user_view_output_file_field', array( $this, 'output_single_user_view_file_field' ), 10, 2 );

	}

	/**
	 * Handle File Upload when admin deletes user's uploaded file.
	 *
	 * @param int $user_id User ID.
	 * @param mixed $profile Profile.
	 */
	public function user_registration_after_admin_save_profile_validation( $user_id, $profile ) {
		$form_id      = ur_get_form_id_by_userid( $user_id );
		$post_content = ur_get_post_content( $form_id );
		$files        = ur_get_form_data_by_key( $post_content, 'file');

		foreach ($files as $field_name => $value) {
			$field_name     = 'user_registration_' . $field_name;
			$previous_files = get_user_meta( $user_id, $field_name );
			$previous_files = isset( $previous_files[0] ) ? explode(",", $previous_files[0] ) : array();
			$current_files  = isset( $_POST[$field_name] ) ? explode( ",", wp_unslash( $_POST[$field_name] ) ) : array();

			if ( ! empty( $previous_files ) ) {
				foreach ( $previous_files as $key => $attachment_id ) {

					if ( ! in_array( $attachment_id, $current_files ) ) {
						$file = get_attached_file( $attachment_id );

						if ( $file ) {
							unlink( $file );
							wp_delete_attachment( $attachment_id, true );
						}
					}
				}
			}
		}
	}

	/**
	 * Handle redirects to setup/welcome page after install and updates.
	 *
	 * For setup wizard, transient must be present, the user must have access rights, and we must ignore the network/bulk plugin updaters.
	 */
	public function admin_redirects() {
		// Setup wizard redirect.
		if ( get_transient( '_urfu_activation_redirect' ) ) {
			delete_transient( '_urfu_activation_redirect' );

			if ( ( is_network_admin() || isset( $_GET['activate-multi'] ) ) || ! current_user_can( 'manage_options' ) || apply_filters( 'urfu_prevent_activation_redirect', false ) ) {
				return;
			}

			$upload_path = UR_UPLOAD_PATH . 'file-uploads';

			if ( ! is_dir( $upload_path ) ) {
				mkdir( $upload_path, 0777, true );
			}

			$message = urfu_is_compatible();
			if ( 'YES' == $message ) {
				// If the user needs to install, send them to the settings page.
				wp_safe_redirect( admin_url( 'admin.php?page=user-registration-settings&tab=file_upload' ) );
				exit;
			}
		}
	}

	/**
	 * Download all uploaded files.
	 *
	 * @return bool
	 */
	public function download_upload_data() {

		$nonce = isset( $_POST['_wpnonce'] ) ? sanitize_text_field( wp_unslash( $_POST['_wpnonce'] ) ) : '';

		$flag = wp_verify_nonce( $nonce, 'user-registration-settings' );

		if ( true != $flag || is_wp_error( $flag ) ) {

			return;
		}

		$form_ids = isset( $_POST['user_registration_file_upload_form_id_for_attachment_download'] ) ? $_POST['user_registration_file_upload_form_id_for_attachment_download'] : array();

		if ( count( $form_ids ) === 0 ) {
			return;
		}

		URFU_Uploaded_Data::render_data( $form_ids );

		$status = URFU_Uploaded_Data::make_zip();

		if ( ! $status ) {

			add_action( 'admin_notices', array( $this, 'user_registration_file_upload_file_download_notices' ), 10 );

			return false;
		}

		$upload_dir = wp_get_upload_dir();

		$archive_file_name = $upload_dir['path'] . DIRECTORY_SEPARATOR . 'user_registration.zip';

		if ( ! file_exists( $archive_file_name ) ) {

			return false;
		}

		header( 'Content-type: application/zip' );
		header( 'Content-Disposition: attachment; filename=user_registration_' . date( 'Y_m_d_H_i_s' ) . '.zip' );
		header( 'Content-length: ' . filesize( $archive_file_name ) );
		header( 'Pragma: no-cache' );
		header( 'Expires: 0' );
		readfile( $archive_file_name );
		if ( file_exists( $archive_file_name ) ) {
			unlink( $archive_file_name );
		}

	}

	/**
	 * Notices for file download.
	 */
	public function user_registration_file_upload_file_download_notices() {

		if ( ! class_exists( 'ZipArchive' ) ) {

			$message = __( 'Please enable <code>ZipArchive</code> extension or please contact with your hosting provider.', 'user-registration-file-upload' );

		} else {

			$message = __( 'Attachment not found.', 'user-registration-file-upload' );
		}
		echo '<div class="notice notice-error"><p>' . esc_html( $message ) . '</p></div>';
	}

	/**
	 * Deprecate plugin missing notice.
	 *
	 * @deprecated 1.1.2
	 *
	 * @return void
	 */
	public function urfu_file_download_notices() {
		ur_deprecated_function( 'urfu_file_download_notices', '1.1.2', 'user_registration_file_upload_file_download_notices' );
	}

	/**
	 * @param $data
	 */
	public function user_registration_profile_field_filter( $data ) {

		$prefix = 'user_registration_';

		if ( isset( $data['fields'] ) ) {

			$data['fields'][ $prefix . $data['field_name'] ]['type'] = 'file';

		}

		return $data;
	}

	/**
	 * Print file upload field in frontend.
	 *
	 * @param $field File upload field.
	 */
	public function user_registration_profile_field( $field ) {

		extract( $field );
		?>
		<button type="button" class="button wp_urfu_upload hide-if-no-js"><?php echo esc_html__( 'Upload file', 'user-registration-file-upload' ); ?></button>
		<div class="urfu-file-uploaded">
			<?php
			$attachment_ids = explode( ',', $value );

			if ( empty( $url ) ) {
				$url = home_url() . '/wp-includes/images/media/text.png';
			}

			if ( ! empty( $value ) ) {
				foreach ( $attachment_ids as $attachment_id ) {
					$attachment_url = wp_get_attachment_url( $attachment_id );
					$file_name      = basename( get_attached_file( $attachment_id ) );
					$file_size      = size_format( filesize( get_attached_file( $attachment_id ) ) );
					?>

					<div class="urfu-file-details">
						<div class="urfu-file-thumb">
							<img src="<?php echo esc_url( $attachment_url ); ?>" />
						</div>

						<div class="urfu-file-heading">
							<h4 class="urfu-file-name" title="<?php echo esc_html( $file_name ); ?> "><?php echo esc_html( $file_name ); ?></h4>
							<span class="urfu-file-size"><?php echo esc_html( $file_size ); ?></span>
						</div>

						<div class="urfu-file-action">
							<a href="<?php echo esc_url( $attachment_url ); ?>" class="urfu-download" target="_blank">
								<span class="dashicons dashicons-download" title="Download"></span>
							</a>

							<a class="urfu-remove" data-attachment-id="<?php echo esc_attr( $attachment_id ); ?>">
								<span class="dashicons dashicons-trash" title="Delete"></span>
							</a>
						</div>
					</div>
					<?php
				}
			}
			?>
		</div>
		<input type="hidden" name="<?php echo esc_attr( $key ); ?>"
			id="<?php echo esc_attr( $key ); ?>"
			value="<?php echo esc_attr( $value ); ?>"
			class="<?php echo( ! empty( $field['class'] ) ? esc_attr( $field['class'] ) : 'regular-text' ); ?> urfu-file-upload-input" />
		<?php
	}

	/**
	 * File upload field ui in form builder.
	 *
	 * @param string $path File path for view.
	 *
	 * @return string
	 */
	public function user_registration_file_admin_template( $path ) {

		$path = URFU_ABSPATH . 'includes/form/views/admin/admin-file.php';

		return $path;

	}


	/**
	 * Add file upload settings page.
	 *
	 * @param array $settings Array of settings pages to include.
	 *
	 * @return array
	 */
	public function add_file_upload_setting( $settings ) {

		if ( class_exists( 'UR_Settings_Page' ) ) {
			$settings[] = include 'settings/class-urfu-settings-file.php';
		}

		return $settings;
	}

	/**
	 * Enqueue scripts
	 */
	public function admin_scripts() {
		$suffix        = defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ? '' : '.min';
		$screen        = get_current_screen();
		$screen_id     = $screen ? $screen->id : '';
		$valid_screens = array( 'profile', 'user-edit' );

		if ( in_array( $screen_id, $valid_screens ) || 'user-registration_page_user-registration-settings' === $screen_id ) {

			wp_enqueue_media();

			urfu_register_upload_script();

			wp_enqueue_script( 'user-registration-file-upload-admin' );

			wp_enqueue_style( 'user-registration-file-upload-admin-style' );

			wp_localize_script(
				'user-registration-file-upload-admin',
				'urfu_upload_script_data',
				array(
					'ajax_url'                => admin_url( 'admin-ajax.php' ),
					'urfu_admin_upload_nonce' => wp_create_nonce( 'urfu_admin_upload_nonce' ),
					'uploader_title'          => __( 'Upload file', 'user-registration-file-upload' ),
					'use_this_file'           => __( 'Use this file', 'user-registration-file-upload' ),
					'urfu_max_file_remaining' => __( 'You can upload up to %qty% files', 'user-registration-file-upload' ),
					'home_url'				  => URFU_PLUGIN_ASSETS,
				)
			);
		}

		wp_register_script(
			'form-builder-file-upload',
			URFU()->plugin_url() . '/assets/js/admin/form-builder-file-upload' . $suffix . '.js',
			array(
				'jquery',
			),
			URFU_VERSION
		);

		if ( 'user-registration_page_add-new-registration' === $screen_id ) {
			wp_enqueue_script( 'form-builder-file-upload' );

			wp_localize_script(
				'form-builder-file-upload',
				'form_builder_file_upload_script_data',
				array(
					'ajax_url'                => admin_url( 'admin-ajax.php' ),
					'urfu_max_file_remaining' => __( 'You can upload up to %qty% files', 'user-registration-file-upload' ),
				)
			);
		}
	}

	/**
	 * Enqueue admin styles.
	 */
	public function admin_styles() {
		$screen        = get_current_screen();
		$tab           = isset( $_GET['tab'] ) ? $_GET['tab'] : '';
		$screen_id     = $screen ? $screen->id : '';
		$valid_screens = array(
			'profile',
			'user-edit',
			'user-registration_page_add-new-registration',
			'user-registration_page_user-registration-users'
		);

		if ( in_array( $screen_id, $valid_screens ) ) {
			wp_register_style( 'user-registration-file-upload-admin-style', URFU()->plugin_url() . '/assets/css/user-registration-file-upload-admin-style.css', array(), URFU_VERSION );
			wp_enqueue_style( 'user-registration-file-upload-admin-style' );

		}

	}


	/**
	 * Add new setting in array general settings of file upload field.
	 *
	 * @param array  $general_settings general field settings.
	 * @param string $id Identifier for file upload field.
	 * @return array
	 */
	public function field_settings( $general_settings, $id ) {

		if ( 'user_registration_file' === $id ) {
			$remove_keys = array( 'placeholder' );

			foreach ( $remove_keys as $remove_key ) {
				unset( $general_settings[ $remove_key ] );
			}

			$new_settings     = array(
				'max_files' => array(
					'setting_id' => 'maximum-number-limit-on-uploads',
					'type'     => 'number',
					'label'    => __( 'Maximum number limit on uploads', 'user-registration-file-upload' ),
					'name'     => 'ur_general_setting[max_files]',
					'required' => true,
					'default'  => 1,
					'tip'      => __( 'Limit the number of files users can upload.', 'user-registration-file-upload' ),
				),
			);
			$general_settings = array_merge( $general_settings, $new_settings );
		}

		return $general_settings;
	}


	/**
	 * File Upload Advance class.
	 *
	 * @param array $file_data File Data.
	 */
	public function user_registration_file_advance_class( $file_data ) {

		$path = URFU_ABSPATH . 'includes/form/settings/class-ur-setting-file.php';

		$file_data['file_path'] = $path;

		return $file_data;

	}

	/**
	 * Output uploaded files for file field in single user view page.
	 *
	 * @param [int] $user_id User Id.
	 * @param [int] $form_id Form Id.
	 * @param [array] $field Field settings.
	 * @return void
	 */
	public function output_single_user_view_file_field( $user_id, $field ) {

		$field_key = $field->general_setting->field_name;

		$uploads = get_user_meta( $user_id, 'user_registration_' . $field_key, true );

		if ( empty( $uploads ) ) {
			echo sprintf(
				"<p class='urfu-no-files-found-message'>%s</p>",
				esc_html__( 'No files uploaded.', 'user-registration' )
			);
			return;
		}

		?>
		<div class="urfu-file-uploaded">
			<?php
			$attachment_ids = explode( ',', $uploads );

			foreach ( $attachment_ids as $attachment_id ) {
				$attachment_url = wp_get_attachment_url( $attachment_id );
				?>

				<div class="urfu-single-user-view-field">
						<div class="urfu-thumbnail">
							<?php
							$attachment_url = wp_get_attachment_url( $attachment_id );

							if ( empty( $attachment_url ) ) {
								$attachment_url = home_url() . '/wp-includes/images/media/text.png';
							}

							$filename_only = basename( get_attached_file( $attachment_id ) ); // Just the file name.
							$filesize_only = size_format( filesize( get_attached_file( $attachment_id ) ), 1 ); // Just the file size.
							$ext           = pathinfo( $attachment_url, PATHINFO_EXTENSION );
							$fileIcon      = user_registration_file_upload_check_file_types( $ext );
							$fileIcon      = ! is_null( $fileIcon ) ? URFU_PLUGIN_ASSETS . '/images/filetypes/' . $fileIcon . '.svg' : $attachment_url;
							?>

							<img data-dz-thumbnail="" alt="<?php echo esc_html( sanitize_text_field( $filename_only ) ); ?>"  src="<?php echo esc_url( $fileIcon ); ?>"/>
						</div>
						<div class="urfu-details">
							<div class="urfu-size">
								<span data-urfu-size=""><strong><?php echo esc_html( sanitize_text_field( $filesize_only ) ); ?></strong></span>
							</div>
							<div class="urfu-filename">
								<span data-urfu-name=""><?php echo esc_html( sanitize_text_field( $filename_only ) ); ?></span>
							</div>
						</div>
						<a class="urfu-download-file" href="<?php echo esc_url( $attachment_url ); ?>" title="Download" target="_blank"><span class="dashicons dashicons-arrow-down-alt"></span></a>
					</div>
				<?php
			}
			?>
		</div>
		<?php
	}

}

return new URFU_Admin();
