<?php
/**
 * Transform class.
 *
 * @package ctpl\classes
 */

namespace CTPL;

use Campo\UserAgent;
use Embed\Http\CurlDispatcher;

/**
 * Trasnformer class.
 */
final class Transform {

	/**
	 * The video source
	 *
	 * @var string $video_source
	 * @access private
	 */
	private $video_source;

	/**
	 * The video id
	 *
	 * @var string $video_id
	 * @access private
	 */
	private $video_id;

	/**
	 * The url of mp4s
	 *
	 * @var array $mp4_urls
	 * @access private
	 */
	private $mp4_urls;

	/**
	 * The poster URL
	 *
	 * @var string $poster_url
	 * @access private
	 */
	private $poster_url;

	/**
	 * Boolean to know if the scrap is done or not
	 *
	 * @var bool $scrap_done
	 * @access private
	 */
	private $scrap_done;

	/**
	 * Construct method.
	 *
	 * @param string $video_source The video source name.
	 * @param string $video_id     The source video ID.
	 *
	 * @throws \Exception If $video_source is not a transformable source.
	 *
	 * @return void
	 */
	public function __construct( $video_source, $video_id ) {
		if ( ! in_array( $video_source, Player::get_transformable_sources(), true ) ) {
			throw new \Exception( $video_source . ' is not transformable.' );
		}
		$this->video_source = (string) $video_source;
		$this->video_id     = (string) $video_id;
		$this->mp4_urls     = array();
		$this->poster_url   = '';
		$this->scrap_done   = false;
	}

	/**
	 * Return the video tag.
	 * $this->mp4_urls must be a not empty array.
	 *
	 * @return mixed string The <video> tag, bool false if $this->mp4_urls is empty.
	 */
	public function get_video_tag() {
		$this->scrap_mp4s_and_poster();
		if ( empty( $this->mp4_urls ) ) {
			return false;
		}
		$video_tag = '<video poster="' . $this->poster_url . '">';
		foreach ( $this->mp4_urls as $video_res => $video_url ) {
			$video_type = Player::get_type_from_video_url( $video_url );
			$hd_attr    = Player::is_hd_resolution( $video_res ) ? 'data-fluid-hd' : '';
			$source_tag = '<source src="' . $video_url . '" type="' . $video_type . '" title="' . $video_res . '"' . $hd_attr . '></source>';
			$video_tag .= $source_tag;
		}
		$video_tag .= '</video>';
		return $video_tag;
	}

	/**
	 * Get mp4 and Poster.
	 *
	 * @access private
	 * @return array The mp4 url(s) and poster url
	 */
	private function scrap_mp4s_and_poster() {
		if ( $this->scrap_done ) {
			return array(
				'mp4_urls'   => $this->mp4_urls,
				'poster_url' => $this->poster_url,
			);
		}
		$this->scrap_done = true;
		switch ( $this->video_source ) {
			case 'redtube':
				return $this->scrap_redtube();
			case 'xvideos':
				return $this->scrap_xvideos();
			case 'youporn':
				return $this->scrap_youporn();
			case 'youporngay':
				return $this->scrap_youporngay();
			default:
		}
		return false;
	}

	/**
	 * Scrap Redtube.
	 *
	 * @return array Redtube mp4s and poster urls
	 */
	private function scrap_redtube() {
		$mp4_urls_by_format = array();
		$mp4_urls           = array();
		$url                = 'https://www.redtube.com/' . $this->video_id;
		$partner_content    = Player::curl( $url, $url );
		$pre_mp4_content    = Player::find_string( 'mediaDefinition:', '],', $partner_content );
		$poster_url         = $this->clear_url( Player::find_string( '"image_url":"', '"', $partner_content ) );

		foreach ( array( 'hls' ) as $format ) {
			$mp4_url_by_format = Player::find_string( $format . '","videoUrl":"', '"', $pre_mp4_content );
			if ( ! empty( $mp4_url_by_format ) ) {
				$mp4_urls_by_format[ $format ] = 'https://redtube.com' . $this->clear_url( $mp4_url_by_format );
			}
		}

		$resolutions = array(
			'1080p' => '1080',
			'720p'  => '720',
			'480p'  => '480',
			'360p'  => '360',
			'240p'  => '240',
		);

		if ( isset( $mp4_urls_by_format['hls'] ) ) {
			$mp4_content = json_decode( Player::curl( $mp4_urls_by_format['hls'], 'https://www.redtube.com/' . $this->video_id ), true );
			if ( ! is_array( $mp4_content ) ) {
				return $this->scrap_redtube_fallback();
			}
			$mp4_url = $mp4_content[0]['videoUrl'];
			foreach ( $resolutions as $key => $resolution ) {
				if ( ! empty( $mp4_url ) ) {
					$mp4_urls[ $key ] = $this->clear_url( $mp4_url );
				}
			}
		}

		if ( 0 === count( $mp4_urls ) && isset( $mp4_urls_by_format['mp4'] ) ) {
			$mp4_content = json_decode( Player::curl( $mp4_urls_by_format['mp4'], 'https://www.redtube.com/' . $this->video_id ), true );
			if ( ! is_array( $mp4_content ) ) {
				return $this->scrap_redtube_fallback();
			}
			foreach ( $mp4_content as $mp4_data ) {
				$mp4_url    = $mp4_data['videoUrl'];
				$resolution = $mp4_data['quality'] . 'p';
				if ( ! empty( $mp4_url ) ) {
					$mp4_urls[ $resolution ] = $this->clear_url( $mp4_url );
				}
			}
		}
		if ( 0 === count( $mp4_urls ) ) {
			return $this->scrap_redtube_fallback();
		}

		$this->mp4_urls   = $mp4_urls;
		$this->poster_url = $this->clear_url( $poster_url );
		return array(
			'mp4_urls'   => $mp4_urls,
			'poster_url' => $poster_url,
		);
	}

	/**
	 * Scrap Redtube Fallback.
	 *
	 * @return array Redtube mp4s and poster urls
	 */
	private function scrap_redtube_fallback() {
		$user_agent = array( 'http' => array( 'user_agent' => 'Mozilla/5.0 (Linux; U; Android 4.0; en-us; GT-I9300 Build/IMM76D)' ) );
		$str        = @file_get_contents( 'https://embed.redtube.com/' . $this->video_id . '/', false, stream_context_create( $user_agent ) );
		if ( false === $str ) {
			return array(
				'mp4_urls'   => array(),
				'poster_url' => '',
			);
		}
		$mp4_content      = Player::find_string( '"videoUrl":"', '"', $str );
		$mp4_urls['480p'] = $this->clear_url( $mp4_content );
		$poster_url       = $this->clear_url( Player::find_string( '"image_url":"', '"', $mp4_content ) );
		$this->mp4_urls   = $mp4_urls;
		$this->poster_url = $this->clear_url( $poster_url );
		return array(
			'mp4_urls'   => $mp4_urls,
			'poster_url' => $poster_url,
		);
	}

	/**
	 * Scrap Xvideos.
	 *
	 * @return array Xvideos mp4s and poster urls
	 */
	private function scrap_xvideos() {
		$mp4_urls        = array();
		$partner_content = Player::curl( 'https://www.xvideos.com/embedframe/' . $this->video_id, 'https://www.xvideos.com' );
		$mp4_content     = Player::find_string( '(html5player) {', '}', $partner_content );
		$poster_url      = Player::find_string( "html5player.setThumbUrl169('", "'", $mp4_content );
		$resolutions     = array(
			'hls'  => 'setVideoHLS',
			'480p' => 'setVideoUrlHigh',
			'360p' => 'setVideoUrlLow',
		);
		foreach ( $resolutions as $key => $resolution ) {
			$mp4_url = Player::find_string( 'html5player.' . $resolution . "('", "');", $mp4_content );
			if ( ! empty( $mp4_url ) ) {
				$mp4_urls[ $key ] = str_replace( 'http:', 'https:', $this->clear_url( $mp4_url ) );
			}
		}
		if ( isset( $mp4_urls['hls'] ) ) {
			unset( $mp4_urls['480p'], $mp4_urls['360p'] );
		}
		$this->mp4_urls   = $mp4_urls;
		$this->poster_url = $this->clear_url( $poster_url );
		return array(
			'mp4_urls'   => $mp4_urls,
			'poster_url' => $poster_url,
		);
	}

	/**
	 * Scrap Youporn.
	 *
	 * @return array Youporn mp4s and poster urls
	 */
	private function scrap_youporn() {
		$mp4_urls    = array();
		$poster_url  = '';
		$mp4_content = json_decode( Player::curl( 'https://www.youporn.com/api/video/media_definitions/' . $this->video_id . '/', 'http://www.youporn.com/watch/' . $this->video_id . '/' ), true );
		foreach ( $mp4_content as $mp4_data ) {
			$mp4_url    = $mp4_data['videoUrl'];
			$resolution = $mp4_data['quality'];
			if ( ! empty( $mp4_url ) ) {
				$mp4_urls[ $resolution ] = $this->clear_url( $mp4_url );
			}
		}
		$this->mp4_urls   = $mp4_urls;
		$this->poster_url = $poster_url;
		return array(
			'mp4_urls'   => $mp4_urls,
			'poster_url' => $poster_url,
		);
	}

	/**
	 * Scrap Youporngay.
	 *
	 * @return array Youporn mp4s and poster urls
	 */
	private function scrap_youporngay() {
		$mp4_urls    = array();
		$poster_url  = '';
		$mp4_content = json_decode( Player::curl( 'https://www.youporngay.com/api/video/media_definitions/' . $this->video_id . '/', 'http://www.youporngay.com/watch/' . $this->video_id . '/' ), true );
		foreach ( $mp4_content as $mp4_data ) {
			$mp4_url    = $mp4_data['videoUrl'];
			$resolution = $mp4_data['quality'];
			if ( ! empty( $mp4_url ) ) {
				$mp4_urls[ $resolution ] = $this->clear_url( $mp4_url );
			}
		}
		$this->mp4_urls   = $mp4_urls;
		$this->poster_url = $poster_url;
		return array(
			'mp4_urls'   => $mp4_urls,
			'poster_url' => $poster_url,
		);
	}

	/**
	 * Just replace'\/' by '/' in a given URL.
	 *
	 * @param string $url The URL to clear.
	 *
	 * @access private
	 * @return string The cleared URL
	 */
	private function clear_url( $url ) {
		return str_replace( '\/', '/', $url );
	}
}
