<?php
/**
 * Plugin Name: WPS Cookie Consent
 * Plugin URI: https://www.wp-script.com/adult-wordpress-plugins/cookie-consent/
 * Description: Displays a notification bar about cookies and GDPR (privacy protection) info.
 * Author: WP-Script
 * Author URI: https://www.wp-script.com
 * Version: 1.0.2
 * Text Domain: coco
 * Domain Path: /languages
 *
 * @package COCO\Main
 */

// Exit if accessed directly.
defined( 'ABSPATH' ) || die( 'Cheatin&#8217; uh?' );

/**
 * Singleton Class.
 *
 * @since 1.0.0
 *
 * @final
 */
final class COCO {

	/**
	 * The instance of WPS COOKIE CONSENT plugin
	 *
	 * @var instanceof WPSCORE $instance
	 * @access private
	 * @static
	 */
	private static $instance;

	/**
	 * The config of WPS COOKIE CONSENT plugin
	 *
	 * @var array $config
	 * @access private
	 * @static
	 */
	private static $config;

	/**
	 * __clone method
	 *
	 * @return void
	 */
	public function __clone() {
		_doing_it_wrong( __FUNCTION__, esc_html__( 'Do not clone or wake up this class', 'coco_lang' ), '1.0' );}

	/**
	 * __wakeup method
	 *
	 * @return void
	 */
	public function __wakeup() {
		_doing_it_wrong( __FUNCTION__, esc_html__( 'Do not clone or wake up this class', 'coco_lang' ), '1.0' );}

	/**
	 * Instance method
	 *
	 * @since 1.0.0
	 *
	 * @return self::$instance
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) && ! ( self::$instance instanceof COCO ) ) {
			include_once ABSPATH . 'wp-admin/includes/plugin.php';

			if ( ! is_plugin_active( 'wp-script-core/wp-script-core.php' ) ) {
				require plugin_dir_path( __FILE__ ) . 'admin/vendors/tgm-activation-x/plugin-activation.php';
				require plugin_dir_path( __FILE__ ) . 'admin/vendors/tgm-activation-x/class-tgm-plugin-activation.php';
			} else {
				self::$instance = new COCO();
				// load config file.
				require plugin_dir_path( __FILE__ ) . 'config.php';
				// load options file.
				require plugin_dir_path( __FILE__ ) . 'admin/pages/page-options-x.php';
				// Load text domain.
				self::$instance->load_textdomain();
				if ( is_admin() ) {
					// Load admin filters.
					self::$instance->load_admin_filters();
					// Load admin hooks.
					self::$instance->load_admin_hooks();
					// Auto-load admin php files.
					self::$instance->auto_load_php_files( 'admin' );
					// Load admin features.
					self::$instance->admin_init();
				}
				if ( ! is_admin() ) {
					// Load public filters.
					// self::$instance->load_public_filters();
					// Load public hooks.
					self::$instance->load_public_hooks();
					// Auto-load admin php files.
					self::$instance->auto_load_php_files( 'public' );
					// Load public features.
					self::$instance->public_init();
				}
			}
		}
		return self::$instance;
	}

	/**
	 * Add js and css files, tabs, pages, php files in admin mode.
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	public function load_admin_filters() {
		add_filter( 'WPSCORE-scripts', array( $this, 'add_admin_scripts' ) );
		add_filter( 'WPSCORE-tabs', array( $this, 'add_admin_navigation' ) );
		add_filter( 'WPSCORE-pages', array( $this, 'add_admin_navigation' ) );
	}

	/**
	 * Add admin js and css scripts. This is a WPSCORE-scripts filter callback function.
	 *
	 * @since 1.0.0
	 *
	 * @param array $scripts List of all WPS CORE CSS / JS to load.
	 * @return array $scripts List of all WPS CORE + WPS COOKIE CONSENT CSS / JS to load.
	 */
	public function add_admin_scripts( $scripts ) {
		if ( isset( self::$config['scripts'] ) ) {
			if ( isset( self::$config['scripts']['js'] ) ) {
				$scripts += (array) self::$config['scripts']['js'];
			}
			if ( isset( self::$config['scripts']['css'] ) ) {
				$scripts += (array) self::$config['scripts']['css'];
			}
		}
		return $scripts;
	}

	/**
	 * Add WPS COOKIE CONSENT admin navigation tab. This is a WPSCORE-tabs and WPSCORE-pages filters callback function.
	 *
	 * @since 1.0.0
	 *
	 * @param array $nav List of all WPS CORE navigation tabs to add.
	 * @return array $nav List of all WPS CORE + WPS COOKIE CONSENT navigation tabs to add.
	 */
	public function add_admin_navigation( $nav ) {
		if ( isset( self::$config['nav'] ) ) {
			// phpcs:disable
			// eval( WPSCORE()->eval_product_data( 'COCO', 'add_admin_navigation' ) );
			$nav += (array) self::$config['nav'];
			// phpcs:enable
		}
		return $nav;
	}

	/**
	 * Auto-loader for PHP files
	 *
	 * @since 1.0.0
	 *
	 * @param string{'admin','public'} $dir Directory where to find PHP files to load.
	 * @static
	 * @return void
	 */
	public static function auto_load_php_files( $dir ) {
		$dirs = (array) ( plugin_dir_path( __FILE__ ) . $dir . '/' );
		foreach ( (array) $dirs as $dir ) {
			$files = new RecursiveIteratorIterator( new RecursiveDirectoryIterator( $dir ) );
			if ( ! empty( $files ) ) {
				foreach ( $files as $file ) {
					// exlude dir.
					if ( $file->isDir() ) {
						continue; }
					// exlude index.php.
					if ( $file->getPathname() === 'index.php' ) {
						continue; }
					// exlude files != .php.
					if ( substr( $file->getPathname(), -4 ) !== '.php' ) {
						continue; }
					// exlude files from -x suffixed directories.
					if ( substr( $file->getPath(), -2 ) === '-x' ) {
						continue; }
					// exlude -x suffixed files.
					if ( substr( $file->getPathname(), -6 ) === '-x.php' ) {
						continue; }
					// else require file.
					require $file->getPathname();
				}
			}
		}
	}

	/**
	 * Registering WPS COOKIE CONSENT activation / deactivation / uninstall hooks.
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	public function load_admin_hooks() {
		register_activation_hook( __FILE__, array( 'COCO', 'activation' ) );
		register_deactivation_hook( __FILE__, array( 'COCO', 'deactivation' ) );
		register_uninstall_hook( __FILE__, array( 'COCO', 'uninstall' ) );
	}

	/**
	 * Stuff to do on WPS COOKIE CONSENT activation. This is a register_activation_hook callback function.
	 *
	 * @since 1.0.0
	 *
	 * @access private
	 * @static
	 * @return void
	 */
	public static function activation() {
		WPSCORE()->update_client_signature();
		WPSCORE()->init( true );
	}

	/**
	 * Stuff to do on WPS COOKIE CONSENT deactivation. This is a register_deactivation_hook callback function.
	 *
	 * @since 1.0.0
	 *
	 * @access private
	 * @static
	 * @return void
	 */
	public static function deactivation() {
		WPSCORE()->update_client_signature();
		WPSCORE()->init( true );
	}

	/**
	 * Stuff to do on WPS COOKIE CONSENT deactivation. This is a register_deactivation_hook callback function.
	 *
	 * @since 1.0.0
	 *
	 * @access private
	 * @static
	 * @return void
	 */
	public static function uninstall() {
		WPSCORE()->update_client_signature();
		WPSCORE()->init( true );
	}

	/**
	 * Text domain function
	 *
	 * @since 1.0.0
	 *
	 * @return false by default
	 */
	public function load_textdomain() {
		// Set filter for plugin's languages directory.
		$lang_dir = dirname( plugin_basename( COCO_FILE ) ) . '/languages/';

		// Traditional WordPress plugin locale filter.
		$mofile = sprintf( '%1$s-%2$s.mo', 'coco_lang', get_locale() );

		// Setup paths to current locale file.
		$mofile_local  = $lang_dir . $mofile;
		$mofile_global = WP_LANG_DIR . '/coco_lang/' . $mofile;

		if ( file_exists( $mofile_global ) ) {
			// Look in global /wp-content/languages/WPSCORE/ folder.
			load_textdomain( 'coco_lang', $mofile_global );
		} elseif ( file_exists( $mofile_local ) ) {
			// Look in local /wp-content/plugins/WPSCORE/languages/ folder.
			load_textdomain( 'coco_lang', $mofile_local );
		} else {
			// Load the default language files.
			load_plugin_textdomain( 'coco_lang', false, $lang_dir );
		}
		return false;
	}

	/**
	 * Load hooks on public mode.
	 *
	 * @since 1.0.0
	 *
	 * @return   void
	 */
	public function load_public_hooks() {
		add_action( 'wp_enqueue_scripts', array( $this, 'load_public_js' ), 100 );
	}

	/**
	 * Load JS files on public mode.
	 *
	 * @since 1.0.0
	 *
	 * @return   void
	 */
	public function load_public_js() {
		// Load VideoJS.
		// phpcs:disable
		wp_enqueue_style( 'coco-cookie-consent-style', plugin_dir_url( __FILE__ ) . 'public/assets/css/cookie-consent.css', array(), COCO_VERSION, 'all' );

		// phpcs:enable
		$cocoDevMode = xbox_get_field_value( 'coco-options', 'cookie-consent-disable-cookie-admin', 'off' );
		if($cocoDevMode == 'on'){
			wp_enqueue_script( 'coco-jquery-cookie', plugin_dir_url( __FILE__ ) . 'public/assets/js/jquery.cookie.min.js', array( 'jquery' ), '1.4.1', true );
		}
		wp_enqueue_script( 'coco-cookie-consent-script', plugin_dir_url( __FILE__ ) . 'public/assets/js/cookie-consent.js', array( 'jquery' ), COCO_VERSION, true );
		wp_enqueue_script( 'coco-cookie-consent-main', plugin_dir_url( __FILE__ ) . 'public/assets/js/main.js', array(), COCO_VERSION, true );
		wp_localize_script(
			'coco-cookie-consent-script',
			'coco_ajax_var',
			array(
				'cookie_consent_disable_cookie_admin'  	=> xbox_get_field_value( 'coco-options', 'cookie-consent-disable-cookie-admin', 'off' ),
				'bar_position'							=> xbox_get_field_value( 'coco-options', 'cookie-consent-bar-position', 'bottom' ),
				'consent_text'                  		=> xbox_get_field_value( 'coco-options', 'cookie-consent-text', 'We use cookies to provide our services. By using this website, you agree to this.' ),
				'consent_button_text'                  	=> xbox_get_field_value( 'coco-options', 'cookie-consent-button', 'Got it!'),
				'display_more_info'						=> xbox_get_field_value( 'coco-options', 'cookie-consent-display-more-information', 'on' ),
				'more_info_text'                        => xbox_get_field_value( 'coco-options', 'cookie-consent-more-information-link-text', 'More information' ),
				'more_info_url'                        	=> xbox_get_field_value( 'coco-options', 'cookie-consent-more-information-link-url', 'https://www.google.com/policies/technologies/cookies/' ),
				'display_privacy'						=> xbox_get_field_value( 'coco-options', 'cookie-consent-display-privacy-protection', 'off' ),
				'privacy_text'                        	=> xbox_get_field_value( 'coco-options', 'cookie-consent-privacy-protection-link-text', 'Privacy protection' ),
				'privacy_popup'                        	=> xbox_get_field_value( 'coco-options', 'cookie-consent-privacy-protection-popup-content', 'Lorem ipsum dolor sit amet, consectetur adipiscing elit, sed do eiusmod tempor incididunt ut labore et dolore magna aliqua. Ut enim ad minim veniam, quis nostrud exercitation ullamco laboris nisi ut aliquip ex ea commodo consequat. Duis aute irure dolor in reprehenderit in voluptate velit esse cillum dolore eu fugiat nulla pariatur. Excepteur sint occaecat cupidatat non proident, sunt in culpa qui officia deserunt mollit anim id est laborum.' ),
				// 'cookie_consent_current_user_can'      	=> current_user_can( 'administrator' ),
			)
		);
		// Load temp style.
		require_once plugin_dir_path( __FILE__ ) . 'dynamic-style.php';
	}

	/**
	 * Stuff to do on admin init.
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	private function admin_init() {}

	/**
	 * Stuff to do on public init.
	 *
	 * @since 1.0.0
	 *
	 * @access private
	 * @return void
	 */
	private function public_init() {}

	/**
	 * Check if current theme is a theme from WP-Script.
	 *
	 * @since 1.1.4
	 *
	 * @return bool true if current theme is a theme from WP-Script, false if not
	 */
	public static function is_wps_theme() {
		$current_theme = wp_get_theme();
		return 'WP-Script' === $current_theme->get( 'Author' );
	}

	/**
	 * Set the list of all terms to translate
	 *
	 * @since 1.0.0
	 *
	 * @return array List of all terms to translate
	 */
	public function get_object_l10n() {
		return array(
			'error_suppression'  => __( 'An error occured during the suppression:', 'coco_lang' ),
			'select_wp_cat'      => __( 'Select a WP category', 'coco_lang' ),
			'select_post_status' => __( 'Select Videos status', 'coco_lang' ),
			'and'                => __( 'AND', 'coco_lang' ),
			'insert_least'       => __( 'Insert at least 1 valid url in the textarea above', 'coco_lang' ),
			'check_least'        => __( 'Check at least 1 video', 'coco_lang' ),
			'enable_button'      => __( 'to enable this button', 'coco_lang' ),
			'import'             => __( 'Import', 'coco_lang' ),
			'search_feed'        => __( 'videos and save this search as a Feed. All your Feeds are displayed at the bottom of this page.', 'coco_lang' ),
		);
	}
}
// End of singleton Class.

/**
 * Create the WPS COOKIE CONSENT instance in a function and call it.
 *
 * @return COCO::instance();
 */
// phpcs:disable
function COCO() {
	return COCO::instance();
}

add_action( 'init', 'coco_check_plugin_status' );
/**
 * Init hook callback to run the plugin only if it is connected.
 */
function coco_check_plugin_status() {
	// Do not run the plugin if it is not connected.
	if ( 'connected' !== WPSCORE()->get_product_status( 'COCO' ) ) {
		return;
	}

	// Do not run the plugin if we are in widget iframe preview.
	if ( isset( $_SERVER['REQUEST_URI'] ) && false !== strpos( $_SERVER['REQUEST_URI'], 'widget' ) ) {
		return;
	}

	COCO();
}
