<?php
/**
 * Main Video class.
 *
 * @package ctpl\classes
 */

namespace CTPL;

/**
 * Main Video class.
 */
final class Video {
	/**
	 * The \DimDom\Document video_tag.
	 *
	 * @var string $tag
	 */
	private $video_tag;

	/**
	 * The post id where the video is.
	 * 0 if the video is not on a post page.
	 * Used to retrieve the post thumbnail to fill the poster if needed.
	 *
	 * @var int $post_id
	 */
	private $post_id;

	/**
	 * A Dom element to easily work with a dom video tag
	 *
	 * @var \DOMDocument $dom
	 */
	private $dom;

	/**
	 * Class constructor.
	 *
	 * @param string $video_tag The video dom tag.
	 * @param int    $post_id   The Post Id where the video_tag comes from.
	 *
	 * @return void
	 */
	public function __construct( $video_tag, $post_id = 0 ) {
		libxml_use_internal_errors( true );
		$this->dom    = new \DOMDocument();
		$content_type = '<meta http-equiv="Content-Type" content="text/html; charset=utf-8">';
		$this->dom->loadHTML( $content_type . $video_tag );
		$this->video_tag = $this->dom->getElementsByTagName( 'video' )->item( 0 );
		libxml_clear_errors();
		$this->post_id = intval( $post_id );
	}

	/**
	 * Get <video> tag string.
	 *
	 * @return string The video tag as a string.
	 */
	public function get_video_tag() {
		return $this->dom->saveHTML( $this->video_tag );
	}

	/**
	 * Add attributes to the tag and render the Player.
	 */
	public function render_player() {
		$this->video_tag->setAttribute( 'poster', $this->get_poster() );
		$this->video_tag->setAttribute( 'id', 'video' );
		$this->video_tag->setAttribute( 'class', '' );
		$this->video_tag->setAttribute( 'width', '100%' );
		$this->video_tag->setAttribute( 'height', '100%' );
		if ( $this->has_multiple_sources() ) {
			$this->update_sources_resolutions();
		}
		if ( $this->is_hls() ) {
			$this->update_hls_source();
		}
		return Player::create_wps_player( $this->get_video_tag(), 'video' );
	}

	/**
	 * Has the current video multiple sources?
	 *
	 * @return bool True if it has multiple sources, false if not.
	 */
	public function has_multiple_sources() {
		$sources = $this->video_tag->getElementsByTagName( 'source' );
		return count( $sources ) > 1;
	}

	/**
	 * Update each source to add title for quality switcher.
	 */
	public function update_sources_resolutions() {
		$sources = $this->video_tag->getElementsByTagName( 'source' );
		foreach ( $sources as $source ) {
			$source_label = $source->getAttribute( 'label' );
			$source_title = $source->getAttribute( 'title' );
			if ( $source_label && ! $source_title ) {
				$source->setAttribute( 'title', $source_label );
			}
		}
	}

	/**
	 * Has the current video any hsl source?
	 *
	 * @return bool True if it has hsl source, false if not.
	 */
	public function is_hls() {
		$sources = $this->video_tag->getElementsByTagName( 'source' );
		foreach ( $sources as $source ) {
			if ( false !== strpos( $source->getAttribute( 'src' ), '.m3u8' ) ) {
				return true;
			}
		}
		return false;
	}

	/**
	 * Update hls source by their resolutions to activate the quality switcher.
	 */
	private function update_hls_source() {
		$sources = $this->video_tag->getElementsByTagName( 'source' );
		foreach ( $sources as $source ) {
			// Exclude Xvideos on mobile.
			if ( false !== strpos( $source->getAttribute( 'src' ), 'xvideos-cdn.com' ) && wp_is_mobile() ) {
				continue;
			}
			// Process if m3u8 is found.
			if ( false !== strpos( $source->getAttribute( 'src' ), '.m3u8' ) ) {
				$source->setAttribute( 'title', 'Auto' );
				$m3u8_from_hls = $this->get_m3u8_from_hls( $source->getAttribute( 'src' ) );
				if ( false !== $m3u8_from_hls ) {
					// Remove old <source> children.
					for ( $i = $sources->length; --$i >= 0; ) {
						$source_to_remove = $sources->item( $i );
						$source_to_remove->parentNode->removeChild( $source_to_remove );
					}
					// Add new <source> children by resolution.
					foreach ( $m3u8_from_hls as $m3u8_res => $m3u8_url ) {
						$source_tag = $this->dom->createElement( 'source' );
						$source_tag->setAttribute( 'src', $m3u8_url );
						$source_tag->setAttribute( 'type', Player::get_type_from_video_url( $m3u8_url ) );
						$source_tag->setAttribute( 'title', $m3u8_res );
						if ( Player::is_hd_resolution( $m3u8_res ) ) {
							$source_tag->setAttribute( 'data-fluid-hd', '' );
						}
						$this->video_tag->appendChild( $source_tag );
					}
				}
				return true;
			}
			return true;
		}
		return true;
	}

	/**
	 * Get m3u8 urls from a given HLS url.
	 *
	 * @param string $hls_url The HLS url.
	 *
	 * @return array List of m3u8 urls indexed with the resolution value (ie. 144/240/320/480/720/1080) if success, bool false if not
	 */
	public function get_m3u8_from_hls( $hls_url ) {
		$parsed_hls_url = wp_parse_url( $hls_url );
		$referer        = $parsed_hls_url['scheme'] . '://' . $parsed_hls_url['host'];
		$base_url_regex = '/^(.+)\//';
		$hls_body       = '';
		preg_match( $base_url_regex, $hls_url, $m3u8_matches, PREG_OFFSET_CAPTURE, 0 );
		$base_url       = $m3u8_matches[0][0];
		$hls_body       = Player::curl( $hls_url, $referer );
		$resolutions    = array();
		$hls_body_regex = '/RESOLUTION=\d+x(\d++),?.*\s(.+\.m3u8(\?.++)??)/mU';
		preg_match_all( $hls_body_regex, $hls_body, $m3u8_matches, PREG_SET_ORDER, 0 );
		foreach ( $m3u8_matches as $m3u8_match ) {
			$m3u8_res                 = $m3u8_match[1];
			$m3u8_file                = $m3u8_match[2];
			$resolutions[ $m3u8_res ] = $base_url . $m3u8_file;
		}
		krsort( $resolutions );
		return ! empty( $resolutions ) ? $resolutions : false;
	}
	/**
	 * Get the poster of the video.
	 *
	 * @return string The poster url.
	 */
	public function get_poster() {
		if ( '' !== $this->video_tag->getAttribute( 'poster' ) ) {
			return $this->video_tag->getAttribute( 'poster' );
		}
		if ( 0 !== $this->post_id && has_post_thumbnail( $this->post_id ) ) {
			return get_the_post_thumbnail_url( $this->post_id );
		}
		if ( 0 !== $this->post_id && get_post_meta( $this->post_id, 'thumb', true ) ) {
			return get_post_meta( $this->post_id, 'thumb', true );
		}
		return '';
	}
}
