<?php
/**
 * WPSCORE Hooks.
 *
 * @api
 * @package CORE\admin\hooks
 */

// Exit if accessed directly.
defined( 'ABSPATH' ) || exit;

/**
 * Filter to add thumb and partner columns when editing posts.
 *
 * @param array $defaults Array of default columns.
 */
function wpscore_add_columns( $defaults ) {
	$defaults['thumb'] = __( 'Thumbnail', 'wpscore_lang' );
	return $defaults;
}
add_filter( 'manage_edit-post_columns', 'wpscore_add_columns' );

/**
 * Action to add thumb and partner content in columns when editig posts.
 *
 * @param string $name The name of the column to add content in.
 */
function wpscore_columns_content( $name ) {
	global $post;
	switch ( $name ) {
		case 'thumb':
			$partner = get_post_meta( $post->ID, 'partner', true );

			$thumb_url = 'https://res.cloudinary.com/themabiz/image/upload/wpscript/sources/admin-no-image.jpg';
			if ( has_post_thumbnail() ) {
				$thumb_url = get_the_post_thumbnail_url( $post->ID, 'wpscript_thumb_admin' );
			} elseif ( get_post_meta( $post->ID, 'thumb', true ) ) {
				$thumb_url = get_post_meta( $post->ID, 'thumb', true );
			}
			echo '<div style="line-height: 0; display: flex; flex-direction: column;">';
			echo wp_kses(
				'<img src="' . esc_url( $thumb_url ) . '" alt="' . esc_attr( get_the_title() ) . '" width="95" height="70" style="width: 100%; height: auto; max-width: 95px;" />',
				wp_kses_allowed_html(
					array(
						'img' => array(
							'alt'    => array(),
							'class'  => array(),
							'height' => array(),
							'src'    => array(),
							'width'  => array(),
							'style'  => array(),
						),
					)
				)
			);

			if ( '' !== $partner ) {
				$partner_img_url = apply_filters( 'wpscore_partner_img_url', 'https://res.cloudinary.com/themabiz/image/upload/wpscript/sources/' . $partner . '.jpg', $partner );
				echo wp_kses(
					'<img src="' . $partner_img_url . '" alt="' . $partner . '" width="95" height="31" style="width: 100%; height: auto; max-width: 95px;"  />',
					wp_kses_allowed_html(
						array(
							'img' => array(
								'alt'    => array(),
								'class'  => array(),
								'height' => array(),
								'src'    => array(),
								'width'  => array(),
								'style'  => array(),
							),
						)
					)
				);
			}
			echo '</div>';
			break;
	}
}
add_action( 'manage_posts_custom_column', 'wpscore_columns_content' );

/**
 * Hook to create admin thumbnails for posts listings.
 */
add_image_size( 'wpscript_thumb_admin', '95', '70', '1' );

/**
 * Update client signature when switching theme.
 *
 * @return void
 */
function wpscore_switch_theme() {
	// update site signature.
	WPSCORE()->update_client_signature();
	// call init.
	WPSCORE()->init( true );
}
add_action( 'after_switch_theme', 'wpscore_switch_theme' );

/**
 * Display admin notice when there are some WP-Script porducts updates.
 *
 * @return bool True if a notice is displayed, false if not.
 */
function wpscript_admin_notice_updates() {
	$is_core_page      = 'toplevel_page_wpscore-dashboard' === get_current_screen()->base ? true : false;
	$available_updates = WPSCORE()->get_available_updates();
	if ( ! current_user_can( 'administrator' ) ) {
		return false;
	}
	if ( 0 === count( $available_updates ) ) {
		return false;
	}
	echo '<div class="notice notice-success is-dismissible">';
	if ( $is_core_page ) {
		echo '<p>Some new WP-Script products versions are available.</p>';
		echo '<p><i class="fa fa-arrow-down" aria-hidden="true"></i> Just <strong>scroll down on this page and press green update buttons</strong> to update products <i class="fa fa-arrow-down" aria-hidden="true"></i></p>';
	} else {
		echo '<p>Some new WP-Script products versions are available: </p>';
		foreach ( $available_updates as $update ) {
			if ( 'CORE' === $update['product_key'] ) {
				$update_url = 'admin.php?page=wpscore-dashboard#wp-script';
				echo '<p>&#10149; ' . esc_html( $update['product_title'] ) . ' <strong>v' . esc_html( $update['product_latest_version'] ) . '</strong> &nbsp;&bull;&nbsp; <a href="' . esc_url( $update_url ) . '">Update</a></p>';
			} else {
				$update_url    = 'admin.php?page=wpscore-dashboard#' . $update['product_key'];
				$changelog_url = 'https://www.wp-script.com/' . $update['product_type'] . '/' . $update['product_slug'] . '/#changelog';
				echo '<p>&#10149; ' . esc_html( $update['product_title'] ) . ' <strong>v' . esc_html( $update['product_latest_version'] ) . '</strong> &nbsp;&bull;&nbsp; <a href="' . esc_url( $update_url ) . '">Update</a> | <a href="' . esc_url( $changelog_url ) . '" target="_blank">Changelog</a></p>';
			}
		}
	}
	echo '</div>';
	return true;
}
add_action( 'admin_notices', 'wpscript_admin_notice_updates' );

/**
 * Add partner filter to posts list.
 *
 * @return void
 */
function wpscript_admin_post_filtering_by_partner_id() {
	/*
	 * Get partners from the WPS plugins.
	 * @var array $partners Array of partners. Default empty array.
	 * Example: array( array( 'id' => 'partner_id', 'name' => 'partner_name' ) ).
	 */
	$partners = apply_filters( 'wpscore_admin_post_filtering_partners', array() );

	// If partners are not an array, return.
	if ( ! is_array( $partners ) ) {
		return;
	}

	// Skip partners that are not well formatted.
	$partners = array_filter(
		$partners,
		function( $partner ) {
			return is_array( $partner ) && isset( $partner['id'] ) && isset( $partner['name'] );
		}
	);

	// If partners are empty, return.
	if ( empty( $partners ) ) {
		return;
	}

	wp_verify_nonce( '_wpnonce' );
	$request_partner = isset( $_REQUEST['partner'] ) ? sanitize_meta( 'partner', wp_unslash( $_REQUEST['partner'] ), 'post' ) : '0';

	echo '<select id="partner" name="partner">';
	echo '<option value="0" ' . ( '0' === $request_partner ? 'selected' : '' ) . '>' . esc_html__( 'All Partners', 'wpscore_lang' ) . ' </option>';
	foreach ( $partners as $partner ) {
		$selected = $request_partner === $partner['id'] ? ' selected="selected"' : '';
		echo '<option value="' . esc_attr( $partner['id'] ) . '"' . esc_attr( $selected ) . '>' . esc_html( $partner['name'] ) . ' </option>';
	}
	echo '</select>';
}
add_action( 'restrict_manage_posts', 'wpscript_admin_post_filtering_by_partner_id', 10 );

/**
 * Filter to add partner filter to posts list.
 *
 * @param WP_Query $query The WP_Query instance (passed by reference).
 * @return WP_Query
 */
function wpscript_query_filter_posts_by_partner_id( $query ) {
	if ( ! ( is_admin() && $query->is_main_query() ) ) {
		return $query;
	}

	wp_verify_nonce( '_wpnonce' );

	if ( ! isset( $_REQUEST['partner'] ) ) {
		return $query;
	}

	$partner_id = sanitize_meta( 'partner', wp_unslash( $_REQUEST['partner'] ), 'post' );

	if ( '0' === $partner_id ) {
		return $query;
	}

	$query->query_vars['meta_key']   = 'partner';
	$query->query_vars['meta_value'] = $partner_id;

	return $query;
}
add_filter( 'parse_query', 'wpscript_query_filter_posts_by_partner_id', 10 );
