<?php
/**
 * Plugin class.
 *
 * @package dvic\classes
 */

// Exit if accessed directly.
defined( 'ABSPATH' ) || exit;

/**
 * DVIC Plugin Singleton Class
 */
final class DVIC {
	/**
	 * The instance of the CORE plugin
	 *
	 * @var      instanceof WPSCORE $instance
	 * @static
	 */
	private static $instance;

	/**
	 * The config of the CORE plugin
	 *
	 * @var      array $config
	 * @static
	 */
	private static $config;

	/**
	 * __clone method
	 *
	 * @return   void
	 */
	public function __clone() {
		_doing_it_wrong( __FUNCTION__, esc_html__( 'Do not clone or wake up this class', 'dvic_lang' ), '1.0' );
	}

	/**
	 * __wakeup method
	 *
	 * @return   void
	 */
	public function __wakeup() {
		_doing_it_wrong( __FUNCTION__, esc_html__( 'Do not clone or wake up this class', 'dvic_lang' ), '1.0' );
	}

	/**
	 * Instance method
	 *
	 * @return   self::$instance
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) && ! ( self::$instance instanceof DVIC ) ) {
			self::$instance = new DVIC();
			// Load textdomain.
			self::$instance->load_textdomain();
			// Load config file.
			include_once DVIC_DIR . 'config.php';
			// Load options file.
			include_once DVIC_DIR . 'admin/pages/page-options-x.php';
			if ( is_admin() ) {
				self::$instance->load_admin_filters();
				self::$instance->load_admin_actions();
				self::$instance->load_admin_hooks();
				self::$instance->auto_load_php_files( 'admin' );
				self::$instance->admin_init();
			}
			if ( ! is_admin() ) {
				self::$instance->public_init();
			}
		}
		return self::$instance;
	}

	/**
	 * Add js and css files, tabs, pages, php files in admin mode.
	 *
	 * @return void.
	 */
	public function load_admin_filters() {
		// add js and css files, tabs, pages, php files.
		add_filter( 'WPSCORE-scripts', array( $this, 'add_admin_scripts' ) );
		add_filter( 'WPSCORE-tabs', array( $this, 'add_admin_navigation' ) );
		add_filter( 'WPSCORE-pages', array( $this, 'add_admin_navigation' ) );
	}

	/**
	 * Add admin js and css scripts. This is a WPSCORE-scripts filter callback function.
	 *
	 * @param array $scripts List of all WPS CORE CSS / JS to load.
	 *
	 * @return array $scripts List of all WPS CORE + plugin CSS / JS to load.
	 */
	public function add_admin_scripts( $scripts ) {
		if ( isset( self::$config['scripts'] ) ) {
			if ( isset( self::$config['scripts']['js'] ) ) {
				$scripts += (array) self::$config['scripts']['js'];
			}
			if ( isset( self::$config['scripts']['css'] ) ) {
				$scripts += (array) self::$config['scripts']['css'];
			}
		}
		return $scripts;
	}

	/**
	 * Add plugin admin navigation tab. This is a WPSCORE-tabs and WPSCORE-pages filters callback function.
	 *
	 * @param array $nav List of all WPS CORE navigation tabs to add.
	 *
	 * @return array $nav List of all WPS CORE + plugin navigation tabs to add.
	 */
	public function add_admin_navigation( $nav ) {
		if ( isset( self::$config['nav'] ) ) {
			$nav += (array) self::$config['nav'];
		}
		return $nav;
	}

	/**
	 * Add admin actions.
	 *
	 * @return void.
	 */
	public function load_admin_actions() {
		add_action( 'deleted_post_meta', array( $this, 'action_deleted_post_meta' ), 10, 4 );
	}

	/**
	 * Delete DVIC_broken_links when deleting DVIC_checked_date.
	 *
	 * @param array  $meta_ids     An array of integers with all meta_ids that are deleted.
	 * @param int    $post_id      The id of the post to which the deleted metas belong.
	 * @param string $meta_key    The meta_key that is deleted.
	 * @param mixed  $meta_values The meta_values that are deleted.
	 *
	 * @return void.
	 */
	public function action_deleted_post_meta( $meta_ids, $post_id, $meta_key, $meta_values ) {
		if ( 'DVIC_checked_date' === $meta_key ) {
			delete_post_meta( $post_id, 'DVIC_broken_links' );
		}
	}

	/**
	 * Auto-loader for PHP files
	 *
	 * @param string{'admin','public'} $dir Directory where to find PHP files to load.
	 *
	 * @return void
	 */
	public function auto_load_php_files( $dir ) {
		$dirs = (array) ( DVIC_DIR . $dir . '/' );
		foreach ( (array) $dirs as $dir ) {
			$files = new RecursiveIteratorIterator( new RecursiveDirectoryIterator( $dir ) );
			if ( ! empty( $files ) ) {
				foreach ( $files as $file ) {
					// exlude dirs.
					if ( $file->isDir() ) {
						continue; }
					// exlude index.php.
					if ( $file->getPathname() === 'index.php' ) {
						continue; }
					// exlude files !== .php.
					if ( substr( $file->getPathname(), -4 ) !== '.php' ) {
						continue; }
					// exlude files from -x suffixed directories.
					if ( substr( $file->getPath(), -2 ) === '-x' ) {
						continue; }
					// exlude -x suffixed files.
					if ( substr( $file->getPathname(), -6 ) === '-x.php' ) {
						continue; }
					// else require file.
					include $file->getPathname();
				}
			}
		}
	}

	/**
	 * Registering plugin activation / deactivation / uninstall hooks.
	 *
	 * @return void
	 */
	public function load_admin_hooks() {
		register_activation_hook( DVIC_FILE, array( __CLASS__, 'activation' ) );
		register_deactivation_hook( DVIC_FILE, array( __CLASS__, 'deactivation' ) );
		register_uninstall_hook( DVIC_FILE, array( __CLASS__, 'uninstall' ) );

	}

	/**
	 * Stuff to do on plugin activation. This is a register_activation_hook callback function.
	 *
	 * @static
	 *
	 * @return void
	 */
	public static function activation() {
		WPSCORE()->update_client_signature();
		WPSCORE()->init( true );
	}

	/**
	 * Stuff to do on plugin deactivation. This is a register_deactivation_hook callback function.
	 *
	 * @static
	 *
	 * @return void
	 */
	public static function deactivation() {
		WPSCORE()->update_client_signature();
		WPSCORE()->init( true );
	}

	/**
	 * Stuff to do on plugin deactivation. This is a register_deactivation_hook callback function.
	 *
	 * @static
	 *
	 * @return void
	 */
	public static function uninstall() {
		WPSCORE()->update_client_signature();
		WPSCORE()->init( true );
	}

	/**
	 * Load textdomain method.
	 *
	 * @return bool True when textdomain is successfully loaded, false if not.
	 */
	public function load_textdomain() {
		$lang = ( current( explode( '_', get_locale() ) ) );
		if ( 'zh' === $lang ) {
			$lang = 'zh-TW';
		}
		$textdomain = 'dvic_lang';
		$mofile     = DVIC_DIR . 'languages/' . $textdomain . '_' . $lang . '.mo';
		return load_textdomain( $textdomain, $mofile );
	}

	/**
	 * Load public filters.
	 *
	 * @return void
	 */
	public function load_public_filters() {
		add_filter( 'WPSCORE-public_dirs', array( $this, 'add_public_dirs' ) );
	}

	/**
	 * Add public php files to require.
	 *
	 * @param array $public_dirs Array of public directories.
	 *
	 * @return array $public_dirs Array of public directories with the current plugin ones.
	 */
	public function add_public_dirs( $public_dirs ) {
		$public_dirs[] = plugin_dir_path( __FILE__ ) . 'public/';
		return $public_dirs;
	}

	/**
	 * Stuff to do on admin init.
	 *
	 * @access private
	 *
	 * @return void
	 */
	private function admin_init() {}

	/**
	 * Stuff to do on public init.
	 *
	 * @access private
	 *
	 * @return void
	 */
	private function public_init() {
		add_action( 'wp_head', array( $this, 'find_broken_links_in_post_background_process' ) );
	}

	/**
	 * Hook to find broken links in a post in the background.
	 */
	public function find_broken_links_in_post_background_process() {
		if ( ! is_singular() ) {
			return;
		}

		$is_bg_process_enabled = 'on' === xbox_get_field_value( 'dvic-options', 'enable-bg-process', 'off' );

		if ( ! $is_bg_process_enabled ) {
			return;
		}

		$last_check      = intval( get_post_meta( get_the_ID(), 'DVIC_checked_date', true ) );
		$daily_frequency = 60 * 60 * 24;
		$frequency       = intval( xbox_get_field_value( 'dvic-options', 'bg-process-frequency', $daily_frequency ) );

		if ( ! $this->must_find_broken_links( $last_check, $frequency ) ) {
			return;
		}

		$only_videos_links            = 'on' === xbox_get_field_value( 'dvic-options', 'bg-process-only-videos-links', 'on' );
		$stop_after_broken_link_found = 'on' === xbox_get_field_value( 'dvic-options', 'bg-process-stop-after-broken-link-found', 'on' );

		$results = DVIC\Checker::find_broken_links_in_post( get_the_ID(), $only_videos_links, $stop_after_broken_link_found );
		if ( ! $results['has_broken_links'] ) {
			return;
		}

		$action = xbox_get_field_value( 'dvic-options', 'bg-process-action', 'report-post' );
		switch ( $action ) {
			case 'report-post':
				break;

			case 'draft-post':
				$post = array(
					'ID'          => get_the_ID(),
					'post_status' => 'draft',
				);
				wp_update_post( $post );
				break;

			default:
				break;
		}
	}

	/**
	 * Check if we must find broken links for the background process.
	 *
	 * @param int $last_check The last time broken links have been checked in seconds.
	 * @param int $frequency  How often the check must be done in seconds.
	 * @return bool True if we must find them, false if not.
	 */
	private function must_find_broken_links( $last_check, $frequency ) {
		return ( time() - $last_check ) > $frequency;
	}
}
