<?php

/**
 * Class VI_WNOTIFICATION_Frontend_Notify
 */
if (!defined('ABSPATH')) {
    exit;
}

class VI_WNOTIFICATION_Frontend_Notify {

    protected $settings;

    protected $lang;
    protected $background_type;

    public function __construct() {
        $this->background_type = 0;
        $this->settings = VI_WNOTIFICATION_Data::get_instance();
        if (isset($_COOKIE['woo_notification_close']) && sanitize_text_field($_COOKIE['woo_notification_close']) && $this->settings->get_params('show_close_icon')) {
            return;
        }
        add_action('wp_print_styles', [$this, 'wp_print_styles']);
        add_action('wp_enqueue_scripts', [$this, 'init_scripts'], 4);

        add_action('wp_ajax_nopriv_woonotification_get_product', [$this, 'product_html']);
        add_action('wp_ajax_woonotification_get_product', [$this, 'product_html']);

        add_action('woocommerce_new_order', [$this, 'update_prefix']);
        add_action('woocommerce_order_status_completed', [$this, 'update_prefix']);
        add_action('woocommerce_order_status_pending', [$this, 'update_prefix']);

        /*Update Recent visited products*/
        if ($this->settings->get_params('archive_page') == 4) {
            add_action('template_redirect', [$this, 'track_product_view'], 21);
        }
    }

    /**
     * Track product views.
     */
    public function track_product_view() {
        if (!is_singular('product')) {
            return;
        }
        if (is_active_widget(false, false, 'woocommerce_recently_viewed_products', true)) {
            return;
        }

        global $post;

        if (empty($_COOKIE['woocommerce_recently_viewed'])) { // @codingStandardsIgnoreLine.
            $viewed_products = [];
        } else {
            $viewed_products = wp_parse_id_list((array)explode('|', wp_unslash($_COOKIE['woocommerce_recently_viewed']))); // @codingStandardsIgnoreLine.
        }

        // Unset if already in viewed products list.
        $keys = array_flip($viewed_products);

        if (isset($keys[$post->ID])) {
            unset($viewed_products[$keys[$post->ID]]);
        }

        $viewed_products[] = $post->ID;

        if (count($viewed_products) > 15) {
            array_shift($viewed_products);
        }
        // Store for session only.
        wc_setcookie('woocommerce_recently_viewed', implode('|', $viewed_products));
    }

    public function update_prefix() {
        $archive_page = $this->settings->get_params('archive_page');
        if (!$archive_page) {
            update_option('_woocommerce_notification_prefix', substr(md5(gmdate("YmdHis")), 0, 10));
        }
    }

    /**
     * Show HTML on front end
     */
    public function product_html() {
        $enable = $this->settings->get_params('enable');

        if ($enable) {
            $products = apply_filters('woonotification_get_products', $this->get_product());
            if (is_array($products) && !empty($products)) {
                echo wp_json_encode($products);
                die;
            } else {
                echo wp_json_encode([]);
                die;
            }
        } else {
            echo wp_json_encode([]);
            die;
        }
    }

    /**
     * @return false|string
     */
    protected function show_product() {
        $image_position = $this->settings->get_params('image_position');
        $position = $this->settings->get_params('position');

        $class = ['wn-background-template-type-' . $this->background_type];
        $class[] = $image_position ? 'img-right' : '';
        $background_image = $this->settings->get_params('background_image');

        switch ($position) {
            case  1:
                $class[] = 'bottom_right';
                break;
            case  2:
                $class[] = 'top_left';
                break;
            case  3:
                $class[] = 'top_right';
                break;
        }
        if ($background_image) {
            $class[] = 'wn-extended';
            $class[] = 'wn-' . $background_image;
        }

        if ($this->settings->enable_rtl()) {
            $class[] = 'wn-rtl';
        }

        if ($this->settings->get_params('rounded_corner')) {
            $class[] = 'wn-rounded-corner';
        }

        ob_start();

        ?>
        <div id="message-purchased" class=" <?php echo esc_attr(implode(' ', $class)) ?>" style="display: none;">

        </div>
        <?php

        return ob_get_clean();
    }

    /**
     * Get virtual names
     *
     * @param int $limit
     *
     * @return array|mixed|void
     */
    public function get_names( $limit = 0 ) {
        $virtual_name = $this->settings->get_params('virtual_name');
        if ($virtual_name && is_string($virtual_name)) {
            $virtual_name = explode("\n", $virtual_name);
            $virtual_name = array_filter($virtual_name);
            if ($limit) {
                if (count($virtual_name) > $limit) {
                    shuffle($virtual_name);

                    return array_slice($virtual_name, 0, $limit);
                }
            }
        }

        return $virtual_name;
    }

    /**
     * Get virtual cities
     *
     * @param int $limit
     *
     * @return array|mixed|void
     */
    public function get_cities( $limit = 0 ) {
        $city = $this->settings->get_params('virtual_city');
        if ($city && is_string($city)) {
            $city = explode("\n", $city);
            $city = array_filter($city, 'trim');
            if ($limit) {
                if (count($city) > $limit) {
                    shuffle($city);

                    return array_slice($city, 0, $limit);
                }
            }
            $city = array_map('trim', $city);
        }

        return $city;
    }

    /**
     * Get all orders given a Product ID.
     *
     * @param integer $product_id The product ID.
     *
     * @return array An array of WC_Order objects.
     * @global        $wpdb
     *
     */
    // phpcs:disable WordPress.DB.PreparedSQL.NotPrepared
    // phpcs:disable WordPress.DB.DirectDatabaseQuery
    protected function get_orders_by_product( $product_id ) {
        if (is_array($product_id)) {
            $product_id = implode(',', $product_id);
        }
        $order_threshold_num = $this->settings->get_params('order_threshold_num');
        $order_threshold_time = $this->settings->get_params('order_threshold_time');
        $order_statuses = $this->settings->get_params('order_statuses');
        if ($order_threshold_num) {
            switch ($order_threshold_time) {
                case 1:
                    $time_type = 'days';
                    break;
                case 2:
                    $time_type = 'minutes';
                    break;
                default:
                    $time_type = 'hours';
            }
            $current_time = strtotime("-" . $order_threshold_num . " " . $time_type);
            $timestamp = gmdate('Y-m-d G:i:s', $current_time);
        }

        global $wpdb;

        $raw = "
        SELECT items.order_id,
          MAX(CASE 
              WHEN itemmeta.meta_key = '_product_id' THEN itemmeta.meta_value
           END) AS product_id
        FROM {$wpdb->prefix}woocommerce_order_items AS items
        INNER JOIN {$wpdb->prefix}woocommerce_order_itemmeta AS itemmeta ON items.order_item_id = itemmeta.order_item_id
        INNER JOIN {$wpdb->prefix}posts AS post ON post.ID = items.order_id
        WHERE items.order_item_type IN('line_item') AND itemmeta.meta_key IN('_product_id','_variation_id') AND post.post_date >= '%s'
        GROUP BY items.order_item_id
        HAVING product_id IN (%s)";

        $sql = $wpdb->prepare($raw, $timestamp, $product_id);
        $results = $wpdb->get_results($sql, ARRAY_A);
        $return = [];
        if (!empty($results)) {
            foreach ($results as $result) {
                $order_id = $result['order_id'];
                $order = wc_get_order($order_id);
                if ($order) {
                    $return[] = $order;
                }
            }
        }

        return $return;
    }
    // phpcs:enable WordPress.DB.PreparedSQL.NotPrepared
    // phpcs:enable WordPress.DB.DirectDatabaseQuery
    /**
     * Process product
     *
     * @return bool
     */
    // phpcs:disable WordPress.DB.SlowDBQuery
    protected function get_product() {
        /* Check nonce if function called from ajax */
        if ((isset($_POST['viwn_pd_id'])
            || isset($_POST['viwn_category_name'])
            && (!isset($_POST['nonce']) || !wp_verify_nonce(sanitize_key($_POST['nonce']), 'viwn_nonce')))) {
            return false;
        }
        $enable_single_product = $this->settings->get_params('enable_single_product');
        $notification_product_show_type = $this->settings->get_params('notification_product_show_type');
        $products = [];
        $product_link = $this->settings->get_params('product_link');
        $product_thumb = $this->settings->get_product_sizes();
        $save_logs = $this->settings->get_params('save_logs');
        $archive_page = $this->settings->get_params('archive_page');
        $product_visibility = $this->settings->get_params('product_visibility') ?: '';
        $prefix = woocommerce_notification_prefix();
        $current_lang = $this->settings::get_current_lang();

        $prefix .= $current_lang;

        /*Check Single Product page*/
        if ($enable_single_product && (is_product() || isset($_POST['viwn_pd_id']))) {
            if (is_product()) {
                $product_id = get_the_ID();
            } else {
                $product_id = isset($_POST['viwn_pd_id']) ? sanitize_text_field($_POST['viwn_pd_id']) : '';
            }

            if (!$product_id) {
                return $products;
            }

            $transient_products = '';//get_transient( $prefix . 'wn_product_child' . $product_id );
            if (is_array($transient_products) && !empty($transient_products)) {
                return $transient_products;
            }
            $product = wc_get_product($product_id);

            /* Only show current product*/
            if (!$notification_product_show_type) {
                /*Show variation products*/
                $enable_variable = $this->settings->get_params('show_variation');
                if ($product->get_type() == 'variable' && $enable_variable) {
                    $temp_p = delete_transient('wn_product_child' . $product_id);

                    if (is_array($temp_p) && !empty($temp_p)) {
                        return $temp_p;
                    } else {
                        $temp_p = $product->get_children();

                        if (!empty($temp_p)) {
                            $rating_html = wc_get_rating_html($product->get_average_rating());
                            $product_thumb = has_post_thumbnail($product_id) ? get_the_post_thumbnail_url($product_id, $product_thumb) : (has_post_thumbnail($product_id) ? get_the_post_thumbnail_url($product_id, $product_thumb) : '');
                            foreach ($temp_p as $key => $the_product) {
                                $product_variation = wc_get_product($the_product);

                                if (!$product_variation->is_in_stock() && !$this->settings->get_params('enable_out_of_stock_product')) {
                                    unset($temp_p[$key]);
                                } else {
                                    if (!empty($product_visibility) && !in_array($product_variation->get_catalog_visibility(), $product_visibility)) {
                                        continue;
                                    }

                                    // do stuff for everything else
                                    $link = $product_variation->get_permalink();
                                    if ($save_logs) {
                                        $link = wp_nonce_url($link, 'wocommerce_notification_click', 'link');
                                    }

                                    $product_tmp = [
                                        'title' => $product_variation->get_name(),
                                        'url' => $link,
                                        'thumb' => $product_thumb,
                                        'rating_html' => $rating_html,
                                        'product_id' => $product_variation->get_id(),
                                        'parent_id' => $product_id,
                                        'price' => $product_variation->get_price('edit'),
                                        'atc_button' => $this->settings->get_button_atc_html($product_variation),
                                    ];

                                    if (!$archive_page) {
                                        $orders = $this->get_orders_by_product($product_id);
                                        if (is_array($orders) && !empty($orders)) {
                                            /** @var \WC_Order $order */
                                            foreach ($orders as $order) {
                                                $order_date_created = $order->get_date_created();
                                                if (!empty($order_date_created)) {
                                                    $order_infor = [
                                                        'time' => $this->time_substract($order_date_created->date_i18n("Y-m-d H:i:s")),
                                                        'time_org' => $order_date_created->date_i18n("Y-m-d H:i:s"),
                                                        'first_name' => base64_encode(ucfirst($order->get_billing_first_name())),
                                                        'last_name' => base64_encode(ucfirst($order->get_billing_last_name())),
                                                        'city' => base64_encode(ucfirst($order->get_billing_city())),
                                                        'state' => base64_encode(ucfirst($order->get_billing_state())),
                                                        'country' => base64_encode(ucfirst(WC()->countries->countries[$order->get_billing_country()])),
                                                    ];

                                                    $products[] = array_merge($product_tmp, $order_infor);
                                                }
                                            }
                                        } else {
                                            continue;
                                        }
                                    } else {
                                        $products[] = $product_tmp;
                                    }
                                }
                            }
                        }
                    }
                } else {
                    if ($product->is_in_stock() || $this->settings->get_params('enable_out_of_stock_product')) {
                        if (!empty($product_visibility) && !in_array($product->get_catalog_visibility(), $product_visibility)) {
                            return false;
                        }

                        // do stuff for everything else
                        $link = $product->get_permalink();
                        if ($save_logs) {
                            $link = wp_nonce_url($link, 'wocommerce_notification_click', 'link');
                        }

                        $product_tmp = [
                            'title' => $product->get_name(),
                            'url' => $link,
                            'thumb' => has_post_thumbnail($product_id) ? get_the_post_thumbnail_url($product_id, $product_thumb) : '',
                            'rating_html' => wc_get_rating_html($product->get_average_rating()),
                            'product_id' => $product->get_id(),
                            'price' => $product->get_price('edit'),
                            'atc_button' => $this->settings->get_button_atc_html($product),
                        ];

                        if (!$archive_page) {
                            $orders = $this->get_orders_by_product($product_id);
                            if (is_array($orders) && !empty($orders)) {
                                /** @var \WC_Order $order */
                                foreach ($orders as $order) {
                                    $order_date_created = $order->get_date_created();
                                    if (!empty($order_date_created)) {
                                        /*28-12-2024 - check order refund*/
                                        if ( is_a( $order, 'WC_Order_Refund' ) ) {
                                            $order = wc_get_order( $order->get_parent_id() );
                                        }
                                        $order_infor = [
                                            'time' => $this->time_substract($order_date_created->date_i18n("Y-m-d H:i:s")),
                                            'time_org' => $order_date_created->date_i18n("Y-m-d H:i:s"),
                                            'first_name' => base64_encode(ucfirst($order->get_billing_first_name())),
                                            'last_name' => base64_encode(ucfirst($order->get_billing_last_name())),
                                            'city' => base64_encode(ucfirst($order->get_billing_city())),
                                            'state' => base64_encode(ucfirst($order->get_billing_state())),
                                            'country' => base64_encode(ucfirst(WC()->countries->countries[$order->get_billing_country()])),
                                        ];

                                        $products[] = array_merge($product_tmp, $order_infor);
                                    }
                                }
                            } else {
                                return false;
                            }
                        } else {
                            $products[] = $product_tmp;
                        }
                    } else {
                        return false;
                    }
                }
            } else {
                /* Show products in the same category*/
                $cates = $product->get_category_ids();
                $args = [
                    'post_type' => 'product',
                    'post_status' => 'publish',
                    'posts_per_page' => 50,
                    'orderby' => 'rand',
                    'post__not_in' => [$product_id], // phpcs:ignore WordPressVIPMinimum.Performance.WPQueryParams.PostNotIn_post__not_in
                    'tax_query' => [
                        [
                            'taxonomy' => 'product_cat',
                            'field' => 'id',
                            'terms' => $cates,
                            'include_children' => false,
                            'operator' => 'IN',
                        ],
                    ],
                    'meta_query' => [
                        [
                            'key' => '_stock_status',
                            'value' => 'instock',
                            'compare' => '=',
                        ],
                    ],
                ];
                if ($this->settings->get_params('enable_out_of_stock_product')) {
                    unset($args['meta_query']);
                }
                $the_query = new WP_Query($args);

                if ($the_query->have_posts()) {
                    while ($the_query->have_posts()) {
                        $the_query->the_post();
                        $same_cate_product_id = get_the_ID();
                        $same_cate_product = wc_get_product($same_cate_product_id);
                        if (!empty($product_visibility) && !in_array($same_cate_product->get_catalog_visibility(), $product_visibility)) {
                            continue;
                        }
                        if ($same_cate_product->is_type('external') && $product_link) {
                            // do stuff for simple products
                            $link = get_post_meta($same_cate_product_id, '_product_url', '#');
                            if (!$link) {
                                $link = get_the_permalink();
                                if ($save_logs) {
                                    $link = wp_nonce_url($link, 'wocommerce_notification_click', 'link');
                                }
                            }
                        } else {
                            // do stuff for everything else
                            $link = get_the_permalink();
                            if ($save_logs) {
                                $link = wp_nonce_url($link, 'wocommerce_notification_click', 'link');
                            }
                        }

                        $product_tmp = [
                            'title' => get_the_title(),
                            'url' => $link,
                            'thumb' => has_post_thumbnail() ? get_the_post_thumbnail_url('', $product_thumb) : '',
                            'rating_html' => wc_get_rating_html($same_cate_product->get_average_rating()),
                            'product_id' => $same_cate_product->get_id(),
                            'price' => $same_cate_product->get_price('edit'),
                            'atc_button' => $this->settings->get_button_atc_html($same_cate_product),
                        ];

                        $products[] = $product_tmp;
                    }
                }

                // Reset Post Data
                wp_reset_postdata();
            }
            if (is_array($products) && !empty($products)) {
                set_transient($prefix . 'wn_product_child' . $product_id, $products, 3600);

                return $products;
            } else {
                return false;
            }
            // Reset Post Data
        }

        /*Get All page*/
        /*Check with Product get from Billing*/
        $limit_product = $this->settings->get_params('limit_product');
        if ($this->settings->get_params('enable_current_category') && (is_product_category() || isset($_POST['viwn_category_name']))) {
            if (is_product_category()) {
                global $wp_query;
                $viwn_category_name = $wp_query->query_vars['product_cat'] ?? '';
            } else {
                $viwn_category_name = isset($_POST['viwn_category_name']) ? sanitize_text_field(wp_unslash($_POST['viwn_category_name'])) : '';
            }
        }

        if ($archive_page > 0) {
            $products = get_transient($prefix . ($viwn_category_name ?? ''));
            if (is_array($products) && !empty($products)) {
                return $products;
            } else {
                $products = [];
            }
            switch ($archive_page) {
                case 1:
                    /*Select Products*/ /*Params from Settings*/ $archive_products = $this->settings->get_params('archive_products');
                    $archive_products = is_array($archive_products) ? $archive_products : [];
                    if (count(array_filter($archive_products)) < 1) {
                        $args = [
                            'post_type' => 'product',
                            'post_status' => 'publish',
                            'posts_per_page' => '50',
                            'orderby' => 'rand',
                            'meta_query' => [
                                [
                                    'key' => '_stock_status',
                                    'value' => 'instock',
                                    'compare' => '=',
                                ],
                            ],
                        ];
                    } else {
                        $args = [
                            'post_type' => ['product', 'product_variation'],
                            'post_status' => 'publish',
                            'posts_per_page' => '50',
                            'orderby' => 'rand',
                            'post__in' => $archive_products,
                            'meta_query' => [
                                [
                                    'key' => '_stock_status',
                                    'value' => 'instock',
                                    'compare' => '=',
                                ],
                            ],

                        ];
                    }
                    break;
                case 2:
                    /*Latest Products*/ /*Params from Settings*/ $args = [
                    'post_type' => 'product',
                    'post_status' => 'publish',
                    'posts_per_page' => $limit_product,
                    'orderby' => 'date',
                    'order' => 'DESC',
                    'meta_query' => [
                        [
                            'key' => '_stock_status',
                            'value' => 'instock',
                            'compare' => '=',
                        ],
                    ],
                ];

                    break;
                case 4:
                    $viewed_products = !empty($_COOKIE['woocommerce_recently_viewed']) ? (array)explode('|', sanitize_text_field(wp_unslash($_COOKIE['woocommerce_recently_viewed']))) : []; // @codingStandardsIgnoreLine
                    $viewed_products = array_reverse(array_filter(array_map('absint', $viewed_products)));

                    if (empty($viewed_products)) {
                        $args = [
                            'post_type' => 'product',
                            'post_status' => 'publish',
                            'posts_per_page' => '50',
                            'orderby' => 'rand',
                            'meta_query' => [
                                [
                                    'key' => '_stock_status',
                                    'value' => 'instock',
                                    'compare' => '=',
                                ],
                            ],

                        ];
                    } else {
                        $args = [
                            'posts_per_page' => $limit_product,
                            'no_found_rows' => 1,
                            'post_status' => 'publish',
                            'post_type' => 'product',
                            'post__in' => $viewed_products,
                            'orderby' => 'post__in',
                            'meta_query' => [
                                [
                                    'key' => '_stock_status',
                                    'value' => 'instock',
                                    'compare' => '=',
                                ],
                            ],
                        ]; // WPCS: slow query ok.
                    }

                    break;
                default:
                    /*Select Categories*/ $cates = $this->settings->get_params('select_categories');
                    if (!empty($cates)) {
                        $categories = get_terms([
                            'taxonomy' => 'product_cat',
                            'include' => $cates,
                        ]);

                        $categories_checked = [];
                        if (!empty($categories)) {
                            foreach ($categories as $category) {
                                $categories_checked[] = $category->term_id;
                            }
                        } else {
                            return false;
                        }

                        /*Params from Settings*/
                        $cate_exclude_products = $this->settings->get_params('cate_exclude_products');

                        if (!is_array($cate_exclude_products)) {
                            $cate_exclude_products = [];
                        }

                        $args = [
                            'post_type' => 'product',
                            'post_status' => 'publish',
                            'posts_per_page' => $limit_product,
                            'orderby' => 'rand',
                            'post__not_in' => $cate_exclude_products, // phpcs:ignore WordPressVIPMinimum.Performance.WPQueryParams.PostNotIn_post__not_in
                            'tax_query' => [
                                [
                                    'taxonomy' => 'product_cat',
                                    'field' => 'id',
                                    'terms' => $categories_checked,
                                    'include_children' => true,
                                    'operator' => 'IN',
                                ],
                            ],
                            'meta_query' => [
                                [
                                    'key' => '_stock_status',
                                    'value' => 'instock',
                                    'compare' => '=',
                                ],
                            ],
                        ];
                    } else {
                        $args = [
                            'post_type' => 'product',
                            'post_status' => 'publish',
                            'posts_per_page' => '50',
                            'orderby' => 'rand',
                            'meta_query' => [
                                [
                                    'key' => '_stock_status',
                                    'value' => 'instock',
                                    'compare' => '=',
                                ],
                            ],

                        ];
                    }
            }
            if (!empty($viwn_category_name)) {
                $args['tax_query'] = [
                    'relation' => 'AND',
                    [
                        'taxonomy' => 'product_cat',
                        'field' => 'slug',
                        'terms' => $viwn_category_name,
                    ],
                ];
            }
            /*Enable in stock*/
            if ($this->settings->get_params('enable_out_of_stock_product')) {
                unset($args['meta_query']);
            }
            if (function_exists('stm_lms_product_remove_from_archive')) {
                remove_action('pre_get_posts', 'stm_lms_product_remove_from_archive');
            }
            $the_query = new WP_Query($args);
            if (function_exists('stm_lms_product_remove_from_archive')) {
                add_action('pre_get_posts', 'stm_lms_product_remove_from_archive');
            }
            if ($the_query->have_posts()) {
                while ($the_query->have_posts()) {
                    $the_query->the_post();
                    $product_id = apply_filters('wpml_object_id', get_the_ID(), 'product', false, $current_lang);
                    $product = wc_get_product($product_id);

                    if (!is_object($product)) {
                        continue;
                    }
                    if (!empty($product_visibility) && !in_array($product->get_catalog_visibility(), $product_visibility)) {
                        continue;
                    }
                    if ($product->is_type('external') && $product_link) {
                        // do stuff for simple products
                        $link = get_post_meta($product_id, '_product_url', '#');
                        if (!$link) {
                            $link = get_the_permalink();
                            if ($save_logs) {
                                $link = wp_nonce_url($link, 'wocommerce_notification_click', 'link');
                            }
                        }
                    } else {
                        // do stuff for everything else
                        $link = get_the_permalink();
                        if ($save_logs) {
                            $link = wp_nonce_url($link, 'wocommerce_notification_click', 'link');
                        }
                    }
                    $product_tmp = [
                        'title' => get_the_title($product_id),
                        'url' => $link,
                        'thumb' => has_post_thumbnail() ? get_the_post_thumbnail_url('', $product_thumb) : '',
                        'rating_html' => wc_get_rating_html($product->get_parent_id() ? (wc_get_product($product->get_parent_id())->get_average_rating()) : $product->get_average_rating()),
                        'product_id' => $product->get_id(),
                        'parent_id' => $product->get_parent_id(),
                        'price' => $product->get_price('edit'),
                        'atc_button' => $this->settings->get_button_atc_html($product),
                    ];
                    if (!$product_tmp['thumb'] && $product->is_type('variation')) {
                        $parent_id = $product->get_parent_id();
                        if ($parent_id) {
                            $product_tmp['thumb'] = has_post_thumbnail($parent_id) ? get_the_post_thumbnail_url($parent_id, $product_thumb) : '';
                        }
                    }
                    $products[] = $product_tmp;
                }
            }
            // Reset Post Data
            wp_reset_postdata();
            if (!empty($products)) {
                set_transient($prefix . ($viwn_category_name ?? ''), $products, 3600);

                return $products;
            } else {
                return false;
            }
        } else {
            /*Get from billing*/
            /*Parram*/
            $order_threshold_num = $this->settings->get_params('order_threshold_num');
            $order_threshold_time = $this->settings->get_params('order_threshold_time');
            $exclude_products = $this->settings->get_params('exclude_products');
            $order_statuses = $this->settings->get_params('order_statuses');

            if (!is_array($exclude_products)) {
                $exclude_products = [];
            }

            $current_time = '';
            if ($order_threshold_num) {
                switch ($order_threshold_time) {
                    case 1:
                        $time_type = 'days';
                        break;
                    case 2:
                        $time_type = 'minutes';
                        break;
                    default:
                        $time_type = 'hours';
                }
                $current_time = strtotime("-" . $order_threshold_num . " " . $time_type);
            }

            $args = [
                'post_type' => 'shop_order',
                'post_status' => $order_statuses,
                'limit' => '100',
                'orderby' => 'date',
                'order' => 'DESC',
            ];

            if ($current_time) {
                if (class_exists('\Automattic\WooCommerce\Utilities\OrderUtil') && \Automattic\WooCommerce\Utilities\OrderUtil::custom_orders_table_usage_is_enabled()) {
                    $args['date_created'] = ">$current_time";
                } else {
                    $args['date_query'] = [
                        [
                            'after' => [
                                'year' => gmdate('Y', $current_time),
                                'month' => gmdate('m', $current_time),
                                'day' => gmdate('d', $current_time),
                                'hour' => gmdate('H', $current_time),
                                'minute' => gmdate('i', $current_time),
                                'second' => gmdate('s', $current_time),
                            ],
                            'inclusive' => true,
                            //(boolean) - For after/before, whether exact value should be matched or not'.
                            //							'compare'   => '<=',
                            //(string) - Possible values are '=', '!=', '>', '>=', '<', '<=', 'LIKE', 'NOT LIKE', 'IN', 'NOT IN', 'BETWEEN', 'NOT BETWEEN', 'EXISTS' (only in WP >= 3.5), and 'NOT EXISTS' (also only in WP >= 3.5). Default value is '='
                            'column' => 'post_date',
                            //(string) - Column to query against. Default: 'post_date'.
                            'relation' => 'AND',
                            //(string) - OR or AND, how the sub-arrays should be compared. Default: AND.
                        ],
                    ];
                }
            }

            if (!empty($viwn_category_name)) {
                $args['tax_query'] = [
                    'relation' => 'AND',
                    [
                        'taxonomy' => 'product_cat',
                        'field' => 'slug',
                        'terms' => $viwn_category_name,
                    ],
                ];
            }

            $orders = wc_get_orders($args);

            $products = [];

            if (!empty($orders)) {
                foreach ($orders as $order) {
                    $items = $order->get_items();
                    foreach ($items as $item) {
                        $line_product_id = $item['product_id'];
                        if (in_array($line_product_id, $exclude_products) || in_array($item['variation_id'], $exclude_products)) {
                            continue;
                        }
                        if ($line_product_id) {
                            //							if ( isset( $item['variation_id'] ) && $item['variation_id'] ) {
                            //								$p_data = wc_get_product( $item['variation_id'] );
                            //							}else{
                            $line_product_id = apply_filters('wpml_object_id', $line_product_id, 'product', false, $current_lang);
                            $p_data = wc_get_product($line_product_id);
                            //                            }
                            if (!is_object($p_data)) {
                                continue;
                            }
                            if (!$p_data->is_in_stock() && !$this->settings->get_params('enable_out_of_stock_product')) {
                                continue;
                            }
                            if ($p_data->get_status() != 'publish') {
                                continue;
                            }
                            if (!empty($product_visibility) && !in_array($p_data->get_catalog_visibility(), $product_visibility)) {
                                continue;
                            }
                            // do stuff for everything else
                            $link = $p_data->get_permalink();
                            if ($save_logs) {
                                $link = wp_nonce_url($link, 'wocommerce_notification_click', 'link');
                            }
                            $line_product_title = get_the_title($line_product_id);
                            if (!empty($item['variation_id'])) {
                                $line_variation_id = apply_filters('wpml_object_id', $item['variation_id'], 'product', false, $current_lang);
                                $line_product_title = get_the_title($line_variation_id);
                            }
                            $product_tmp = [
                                'title' => apply_filters('wcn_product_title', $line_product_title, $line_product_id),
                                'url' => $link,
                                'thumb' => has_post_thumbnail($line_product_id) ? get_the_post_thumbnail_url($line_product_id, $product_thumb) : '',
                                'rating_html' => wc_get_rating_html($p_data->get_parent_id() ? (wc_get_product($p_data->get_parent_id())->get_average_rating()) : $p_data->get_average_rating()),
                                'parent_id' => $p_data->get_parent_id(),
                                'product_id' => $p_data->get_id(),
                                'price' => $p_data->get_price('edit'),
                                'atc_button' => $this->settings->get_button_atc_html($p_data),
                                'time' => $this->time_substract($order->get_date_created()->date_i18n("Y-m-d H:i:s")),
                                'time_org' => $order->get_date_created()->date_i18n("Y-m-d H:i:s"),
                                'first_name' => base64_encode(ucfirst($order->get_billing_first_name())),
                                'last_name' => base64_encode(ucfirst($order->get_billing_last_name())),
                                'city' => base64_encode(ucfirst($order->get_billing_city())),
                                'state' => base64_encode(ucfirst($order->get_billing_state())),
                                'country' => base64_encode(ucfirst(WC()->countries->countries[$order->get_billing_country()])),
                            ];
                            if (!$product_tmp['thumb'] && $p_data->is_type('variation')) {
                                $parent_id = $p_data->get_parent_id();
                                if ($parent_id) {
                                    $product_tmp['thumb'] = has_post_thumbnail($parent_id) ? get_the_post_thumbnail_url($parent_id, $product_thumb) : '';
                                }
                            }

                            $products[] = $product_tmp;
                        }
                    }
                    $products = array_map("unserialize", array_unique(array_map("serialize", $products)));
                    $products = array_values($products);
                    if (count($products) >= 100) {
                        break;
                    }
                }
            }

            if (!empty($products)) {
                return $products;
            } else {
                return false;
            }
        }
    }
    // phpcs:enable WordPress.DB.SlowDBQuery.

    /**
     * Get time
     *
     * @param      $time
     * @param bool $number
     * @param bool $calculate
     *
     * @return bool|string
     */
    protected function time_substract( $time, $number = false, $calculate = false ) {
        if (!$number) {
            if ($time) {
                $time = strtotime($time);
            } else {
                return false;
            }
        }

        if (!$calculate) {
            $current_time = current_time('timestamp');
            //			echo "$current_time - $time";
            $time_substract = $current_time - $time;
        } else {
            $time_substract = $time;
        }
        if ($time_substract > 0) {
            /*Check day*/
            $day = $time_substract / (24 * 3600);
            $day = intval($day);
            if ($day > 1) {
                return $day . ' ' . esc_html__('days', 'woocommerce-notification');
            } elseif ($day > 0) {
                return $day . ' ' . esc_html__('day', 'woocommerce-notification');
            }

            /*Check hour*/
            $hour = $time_substract / (3600);
            $hour = intval($hour);
            if ($hour > 1) {
                return $hour . ' ' . esc_html__('hours', 'woocommerce-notification');
            } elseif ($hour > 0) {
                return $hour . ' ' . esc_html__('hour', 'woocommerce-notification');
            }

            /*Check min*/
            $min = $time_substract / (60);
            $min = intval($min);
            if ($min > 1) {
                return $min . ' ' . esc_html__('minutes', 'woocommerce-notification');
            } elseif ($min > 0) {
                return $min . ' ' . esc_html__('minute', 'woocommerce-notification');
            }

            return intval($time_substract) . ' ' . esc_html__('seconds', 'woocommerce-notification');
        } else {
            return esc_html__('a few seconds', 'woocommerce-notification');
        }
    }

    /**
     *
     * @return mixed
     */
    protected function get_custom_shortcode() {
        $message_shortcode = $this->settings->get_params('custom_shortcode');
        $min_number = $this->settings->get_params('min_number');
        $max_number = $this->settings->get_params('max_number');

        $number = wp_rand($min_number, $max_number);
        $message = preg_replace('/\{number\}/i', $number, $message_shortcode);

        return $message;
    }

    /**Deprecated
     * check woocommerce-notification.js
     *
     * @return string
     */
    protected function message_purchased() {
        $message_purchased = $this->settings->get_params('message_purchased');
        $show_close_icon = $this->settings->get_params('show_close_icon');
        $archive_page = $this->settings->get_params('archive_page');
        $product_link = $this->settings->get_params('product_link');
        $image_redirect = $this->settings->get_params('image_redirect');
        $image_redirect_target = $this->settings->get_params('image_redirect_target');
        if (is_array($message_purchased)) {
            $index = wp_rand(0, count($message_purchased) - 1);
            $message_purchased = $message_purchased[$index];
        }
        $messsage = '';
        $keys = [
            '{first_name}',
            '{last_name}',
            '{city}',
            '{state}',
            '{country}',
            '{product}',
            '{product_with_link}',
            '{time_ago}',
            '{custom}',
        ];

        $product = $this->get_product();

        if ($product) {
            $product_id = $product['id'];
        } else {
            return false;
        }

        $first_name = trim($product['first_name']);
        $last_name = trim($product['last_name']);

        $city = trim($product['city']);
        $state = trim($product['state']);
        $country = trim($product['country']);
        $time = trim($product['time']);
        if (!$archive_page) {
            $time = $this->time_substract($time);
        }

        $_product = wc_get_product($product_id);
        if (woocommerce_version_check()) {
            $product = '<span class="wn-popup-product-title">' . esc_html(wp_strip_all_tags(get_the_title($product_id))) . '</span>';
        } else {
            $prd_var_title = $_product->post->post_title;
            if ($_product->get_type() == 'variation') {
                $prd_var_attr = $_product->get_variation_attributes();
                $attr_name1 = array_values($prd_var_attr)[0];
                $product = $prd_var_title . ' - ' . $attr_name1;
            } else {
                $product = $prd_var_title;
            }

            $product = wp_strip_all_tags($product);
        }

        if ($_product->is_type('external') && $product_link) {
            // do stuff for simple products
            $link = get_post_meta($product_id, '_product_url', '#');
            if (!$link) {
                $link = get_permalink($product_id);
                $link = wp_nonce_url($link, 'wocommerce_notification_click', 'link');
            }
        } else {
            // do stuff for everything else
            $link = get_permalink($product_id);
            $link = wp_nonce_url($link, 'wocommerce_notification_click', 'link');
        }
        ob_start();
        ?>
        <a <?php if ($image_redirect_target) {
            echo 'target="_blank"';
        } ?> href="<?php echo esc_url($link) ?>"><?php echo esc_html($product) ?></a>
        <?php
        $product_with_link = ob_get_clean();

        ob_start();
        ?>
        <small><?php echo esc_html__('About', 'woocommerce-notification') . ' ' . esc_html($time) . ' ' . esc_html__('ago', 'woocommerce-notification') ?></small>
        <?php
        $time_ago = ob_get_clean();
        $product_thumb = $this->settings->get_product_sizes();
        if (has_post_thumbnail($product_id)) {
            if ($image_redirect) {
                $messsage .= '<a ' . ($image_redirect_target ? 'target="_blank"' : '') . ' href="' . esc_url($link) . '">';
                $messsage .= '<img src="' . esc_url(get_the_post_thumbnail_url($product_id, $product_thumb)) . '" class="wcn-product-image"/>';
                $messsage .= '</a>';
            } else {
                $messsage .= '<img src="' . esc_url(get_the_post_thumbnail_url($product_id, $product_thumb)) . '" class="wcn-product-image"/>';
            }
        } elseif ($_product->get_type() == 'variation') {
            $parent_id = $_product->get_parent_id();
            if ($image_redirect && $parent_id) {
                $messsage .= '<a ' . ($image_redirect_target ? 'target="_blank"' : '') . ' href="' . esc_url($link) . '">';
                $messsage .= '<img src="' . esc_url(get_the_post_thumbnail_url($parent_id, $product_thumb)) . '" class="wcn-product-image"/>';
                $messsage .= '</a>';
            } else {
                $messsage .= '<img src="' . esc_url(get_the_post_thumbnail_url($parent_id, $product_thumb)) . '" class="wcn-product-image"/>';
            }
        }

        //Get custom shortcode
        $custom_shortcode = $this->get_custom_shortcode();
        $replaced = [
            $first_name,
            $last_name,
            $city,
            $state,
            $country,
            $product,
            $product_with_link,
            $time_ago,
            $custom_shortcode,
        ];
        $messsage .= str_replace($keys, $replaced, '<p>' . wp_strip_all_tags($message_purchased) . '</p>');
        ob_start();
        if ($show_close_icon) {
            ?>
            <span id="notify-close"></span>
            <?php
        }
        $messsage .= ob_get_clean();

        return $messsage;
    }

    /**
     * Show HTML code
     */
    public function wp_footer() {
        if (!is_admin()) {
            $sound_enable = $this->settings->get_params('sound_enable');
            $sound = $this->settings->get_params('sound');

            echo $this->show_product(); // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped

            if ($sound_enable) {
                ?>
                <audio id="woocommerce-notification-audio">
                    <source src="<?php echo esc_url(VI_WNOTIFICATION_SOUNDS_URL . $sound) ?>">
                </audio>
                <?php
            }
        }
    }

    /**
     * Add Script and Style
     */
    public function wp_print_styles() {
        ?>
        <link id="woocommerce-notification-fonts" rel="preload" href="<?php echo esc_attr(esc_url(VI_WNOTIFICATION_FONT . 'icons-close.woff2')) ?>" as="font" crossorigin>
        <?php
    }

    function init_scripts() {
        $enable = $this->settings->get_params('enable');
        if (!$enable) {
            return;
        }

        $enable_mobile = $this->settings->get_params('enable_mobile');
        if (!$enable_mobile && wp_is_mobile()) {
            return;
        }
        // Any mobile device (phones or tablets).
        $mobile_detect = new VillaTheme_Mobile_Detect();
        if (!$enable_mobile && $mobile_detect->isMobile()) {
            return;
        }
        $is_home = $this->settings->get_params('is_home');
        $is_checkout = $this->settings->get_params('is_checkout');
        $is_cart = $this->settings->get_params('is_cart');
        $prefix = woocommerce_notification_prefix();
        /*Conditional tags*/
        $logic_value = $this->settings->get_params('conditional_tags');
        /*Assign page*/
        if ($is_home && (is_home() || is_front_page())) {
            return;
        }
        if ($is_checkout && is_checkout()) {
            return;
        }
        if ($is_cart && is_cart()) {
            return;
        }
        if ($logic_value) {
            if (stristr($logic_value, "return") === false) {
                $logic_value = "return (" . $logic_value . ");";
            }
            if (!eval($logic_value)) { // phpcs:ignore Generic.PHP.ForbiddenFunctions.Found
                return;
            }
        }
        $this->lang = $this->settings::get_current_lang();
        $options_array = get_transient($prefix . '_head' . $this->lang);
        $non_ajax = $this->settings->get_params('non_ajax');
        $archive = $this->settings->get_params('archive_page');
        $enable_single_product = $this->settings->get_params('enable_single_product');
        $notification_product_show_type = $this->settings->get_params('notification_product_show_type');
        /*Autodetect*/
        $detect = $this->settings->get_params('country');
        if (!is_array($options_array) || empty($options_array)) {
            $options_array = [];
            $loop = $this->settings->get_params('loop');
            $options_array['loop'] = $loop;
            $loop_session = $this->settings->get_params('loop_session');
            $options_array['loop_session'] = $loop_session;
            $loop_session_total = $this->settings->get_params('loop_session_total');
            $options_array['loop_session_total'] = $loop_session_total;
            $options_array['close_icon_style'] = $this->settings->get_params('close_icon_style', 0);
            $loop_session_duration = $this->settings->get_params('loop_session_duration');
            $loop_session_duration_unit = $this->settings->get_params('loop_session_duration_unit');
            switch ($loop_session_duration_unit) {
                case 'h':
                    $loop_session_duration = $loop_session_duration * 3600;
                    break;
                case 'm':
                    $loop_session_duration = $loop_session_duration * 60;
                    break;
                default:
            }
            $options_array['loop_session_duration'] = $loop_session_duration;

            $display_time = $this->settings->get_params('display_time');
            $options_array['display_time'] = $display_time;

            $next_time = $this->settings->get_params('next_time');
            $options_array['next_time'] = $next_time;

            $notification_per_page = $this->settings->get_params('notification_per_page');
            $options_array['notification_per_page'] = $notification_per_page;

            $message_display_effect = $this->settings->get_params('message_display_effect');
            $options_array['display_effect'] = $message_display_effect;

            $message_hidden_effect = $this->settings->get_params('message_hidden_effect');
            $options_array['hidden_effect'] = $message_hidden_effect;

            $target = $this->settings->get_params('image_redirect_target');
            $options_array['redirect_target'] = $target;

            $image_redirect = $this->settings->get_params('image_redirect');
            $options_array['image'] = $image_redirect;

            $message_purchased = $this->settings->get_params('message_purchased');
            if (!is_array($message_purchased)) {
                $message_purchased = [$message_purchased];
            }
            $options_array['messages'] = $message_purchased;
            $options_array['message_custom'] = $this->settings->get_params('custom_shortcode');
            $options_array['message_number_min'] = $this->settings->get_params('min_number');
            $options_array['message_number_max'] = $this->settings->get_params('max_number');

            $options_array['detect'] = $detect;

            /*Check get from billing*/
            /*Current products*/
            if ($archive || ($notification_product_show_type && is_product() && $enable_single_product)) {
                $virtual_time = $this->settings->get_params('virtual_time');
                $options_array['time'] = $virtual_time;
                $names = $this->get_names(50);
                if (is_array($names) && !empty($names)) {
                    $options_array['names'] = array_map('base64_encode', $names);
                }
                if ($detect == 1) {
                    $cities = $this->get_cities(50);
                    if (is_array($cities) && !empty($cities)) {
                        $options_array['cities'] = array_map('base64_encode', $cities);
                    }
                    $options_array['country'] = $this->settings->get_params('virtual_country');
                }
            } else {
                if (!$non_ajax && !is_product()) {
                    $options_array['ajax_url'] = admin_url('admin-ajax.php');
                }
            }
            set_transient($prefix . '_head' . $this->lang, $options_array, 86400);
        }
        $wn_ip = isset($_COOKIE['wn_ip']) ? sanitize_text_field($_COOKIE['wn_ip']) : '';
        if (empty($wn_ip)) {
            try {
                $request_ip = wp_remote_get('http://ip-api.com/json/', []);
                $body_ip = json_decode(wp_remote_retrieve_body($request_ip), true);
                if (!empty($body_ip)) {
                    setcookie('wn_ip', $body_ip['query'] ?? '', time() + 84600 * 10);
                    if (isset($body_ip['status']) && $body_ip['status'] == 'success') {
                        setcookie('wn_city', $body_ip['city'] ?? '', time() + 84600 * 10);
                        setcookie('wn_country', $body_ip['country'] ?? '', time() + 84600 * 10);
                    }
                }
            } catch (Exception $e) {

            }
        }
        add_action('wp_footer', [$this, 'wp_footer']);

        if (WP_DEBUG) {
            wp_enqueue_style('woocommerce-notification-icons-close', VI_WNOTIFICATION_CSS . 'icons-close.css', [], VI_WNOTIFICATION_VERSION);
            wp_enqueue_style('woocommerce-notification', VI_WNOTIFICATION_CSS . 'woocommerce-notification.css', [], VI_WNOTIFICATION_VERSION);
            wp_enqueue_script('woocommerce-notification', VI_WNOTIFICATION_JS . 'woocommerce-notification.js', ['jquery'], VI_WNOTIFICATION_VERSION, true);
        } else {
            wp_enqueue_style('woocommerce-notification-icons-close', VI_WNOTIFICATION_CSS . 'icons-close.min.css', [], VI_WNOTIFICATION_VERSION);
            wp_enqueue_style('woocommerce-notification', VI_WNOTIFICATION_CSS . 'woocommerce-notification.min.css', [], VI_WNOTIFICATION_VERSION);
            wp_enqueue_script('woocommerce-notification', VI_WNOTIFICATION_JS . 'woocommerce-notification.min.js', ['jquery'], VI_WNOTIFICATION_VERSION, true);
        }

        if ($this->settings->get_params('background_image')) {
            wp_enqueue_style('woocommerce-notification-templates', VI_WNOTIFICATION_CSS . 'woocommerce-notification-templates.css', [], VI_WNOTIFICATION_VERSION);
        }

        $options_array = array_merge([
            'str_about' => esc_html__('About', 'woocommerce-notification'),
            'str_ago' => esc_html__('ago', 'woocommerce-notification'),
            'str_day' => esc_html__('day', 'woocommerce-notification'),
            'str_days' => esc_html__('days', 'woocommerce-notification'),
            'str_hour' => esc_html__('hour', 'woocommerce-notification'),
            'str_hours' => esc_html__('hours', 'woocommerce-notification'),
            'str_min' => esc_html__('minute', 'woocommerce-notification'),
            'str_mins' => esc_html__('minutes', 'woocommerce-notification'),
            'str_secs' => esc_html__('secs', 'woocommerce-notification'),
            'str_few_sec' => esc_html__('a few seconds', 'woocommerce-notification'),
            'time_close' => $this->settings->get_params('time_close'),
            'show_close' => $this->settings->get_params('show_close_icon'),
            'change_virtual_time' => $this->settings->get_params('change_virtual_time_enable'),
            'change_message_number' => $this->settings->get_params('change_message_number_enable'),
        ], $options_array);

        if ($options_array['detect'] === '2') {
            $ip = new WC_Geolocation();
            $geo_ip = $ip->geolocate_ip('', true, true);

            $names = $this->get_name_by_country($geo_ip['country'] ?? '');

            if (!empty($names)) {
                $options_array['names'] = $names;
            }

            $woo_countries = new WC_Countries();
            $woo_countries = $woo_countries->__get('countries');
            $geo_ip['country'] = $woo_countries[$geo_ip['country']] ?? $geo_ip['country'] ?? '';
            if (empty($geo_ip['city'])) {
                $ip_address = $ip->get_ip_address();
                $geo_city = get_transient('wn_city_' . $ip_address);
                if ($geo_city === false) {
                    $geoip_services = [
                        'ip-api.com' => 'http://ip-api.com/json/%s',
                        'ipinfo.io' => 'https://ipinfo.io/%s/json',
                    ];
                    if (!empty($geoip_services)) {
                        $geoip_services_keys = array_keys($geoip_services);
                        foreach ($geoip_services_keys as $service_name) {
                            if ($geo_city) {
                                break;
                            }
                            $service_endpoint = $geoip_services[$service_name];
                            $response = wp_safe_remote_get(sprintf($service_endpoint, $ip_address), ['timeout' => 2]);
                            if (!is_wp_error($response) && $response['body']) {
                                $data = json_decode($response['body']);
                                $geo_city = $data->city ?? '';
                            }
                        }
                    }
                    set_transient('wn_city_' . $ip_address, $geo_city, DAY_IN_SECONDS);
                }
                $geo_ip['city'] = $geo_city;
            }
            $geo_ip_t = apply_filters('wn_get_geo_ip', $geo_ip);
            $options_array['cities'] = [$geo_ip_t['city'] ?? ''];
            $options_array['country'] = $geo_ip_t['country'] ?? '';
        }

        if ($options_array['change_virtual_time']) {
            $options_array['start_virtual_time'] = $this->get_start_virtual_time();
            $options_array['end_virtual_time'] = $this->get_end_virtual_time();
        }
        if ($options_array['change_message_number']) {
            $options_array['current_hour'] = (int)gmdate('H', current_time('timestamp'));
        }

        if ($notification_product_show_type && is_product() && $enable_single_product) {
            $options_array['in_the_same_cate'] = 1;
            $options_array['billing'] = 0;
        } else {
            $options_array['in_the_same_cate'] = 0;
            $options_array['billing'] = 1;
        }
        if ($archive) {
            $options_array['billing'] = 0;
        } else {
            $options_array['billing'] = 1;
        }
        /*Notification options*/
        $initial_delay = $this->settings->get_params('initial_delay');
        $initial_delay_random = $this->settings->get_params('initial_delay_random');
        if ($initial_delay_random) {
            $initial_delay_min = $this->settings->get_params('initial_delay_min');
            $initial_delay = wp_rand(intval($initial_delay_min), intval($initial_delay));
        }
        $options_array['initial_delay'] = $initial_delay;
        /*Process products, address, time */
        /*Load products*/
        if ($non_ajax && ($archive || is_product())) {
            $options_array['ajax_url'] = '';
            $products = apply_filters('woonotification_get_products', $this->get_product());
        } else {
            $options_array['ajax_url'] = admin_url('admin-ajax.php');
            $options_array['viwn_pd_id'] = $enable_single_product && is_product() ? get_the_ID() : '';
            if ($this->settings->get_params('enable_current_category') && is_product_category()) {
                global $wp_query;
                $options_array['viwn_category_name'] = $wp_query->query_vars['product_cat'] ?? '';
            }
            $products = [];
        }
        if (is_array($products) && !empty($products)) {
            $options_array['products'] = $products;
        }
        $options_array['enableMobile'] = $enable_mobile;
        $options_array['atc_icon_url'] = VI_WNOTIFICATION_IMAGES . 'add-cart.png';
        $options_array['nonce'] = wp_create_nonce('viwn_nonce');
        $options_array['is_single_product'] = is_product();
        wp_localize_script('woocommerce-notification', '_woocommerce_notification_params', $options_array);
        /*Custom*/

        $highlight_color = $this->settings->get_params('highlight_color');
        $text_color = $this->settings->get_params('text_color');
        $background_color = $this->settings->get_params('background_color');
        $image_border_radius = $this->settings->get_params('image_border_radius');
        $custom_css_setting = $this->settings->get_params('custom_css');
        $background_image = $this->settings->get_params('background_image');
        $image_padding = $this->settings->get_params('image_padding');
        $close_icon_color_classic = $this->settings->get_params('close_icon_color_classic');
        $close_icon_color_default = $this->settings->get_params('close_icon_color_default');
        $close_icon_background = $this->settings->get_params('close_icon_background');
        $custom_css = '#notify-close.default:before{color:' . $close_icon_color_default . ';}';
        $custom_css .= '#notify-close.classic:before{color:' . $close_icon_color_classic . ';}';
        $custom_css .= "#message-purchased #notify-close.default{background-color:{$close_icon_background};}";
        $is_rtl = $this->settings->enable_rtl();
        if ($background_image) {
            $border_radius = $this->settings->get_params('border_radius', 0) . 'px';
            $this->background_type = 2;
            $background_image_url = woocommerce_notification_background_images($background_image);

            $custom_css .= "#message-purchased .message-purchase-main::before{background-image: url('{$background_image_url}'); border-radius:{$border_radius};}";
        } else {
            $custom_css .= '#message-purchased .message-purchase-main{overflow:hidden}';
            $border_radius = $this->settings->get_params('border_radius', 0) . 'px';
        }
        if ($image_padding) {
            $padding_right = 15 - $image_padding;
            $custom_css .= "#message-purchased .wn-notification-image-wrapper{padding:{$image_padding}px;}";
            if ($is_rtl) {
                $custom_css .= "#message-purchased .wn-notification-message-container{padding-right:{$padding_right}px}";
            } else {
                $custom_css .= "#message-purchased .wn-notification-message-container{padding-left:{$padding_right}px;}";
            }
            $custom_css .= "#message-purchased .wn-notification-image{border-radius:{$image_border_radius}px;}";
        } else {
            $custom_css .= "#message-purchased .wn-notification-image-wrapper{padding:0;}";
            if ($is_rtl) {
                $custom_css .= "#message-purchased .wn-notification-message-container{padding-right:15px;}";
            } else {
                $custom_css .= "#message-purchased .wn-notification-message-container{padding-left:15px;}";
            }
            $custom_css .= "#message-purchased .wn-notification-image{border-radius:{$image_border_radius}px;}";
        }
        $custom_css .= "#message-purchased .message-purchase-main{background-color: {$background_color};color:{$text_color} !important;border-radius:{$border_radius};}#message-purchased .message-purchase-main .wn-notification-message-container,#message-purchased .message-purchase-main .wn-notification-message-container small,#message-purchased .wn-notification-message-container .wn-notification-atc > i{color:{$text_color} !important;}#message-purchased .message-purchase-main .wn-notification-message-container a, #message-purchased .message-purchase-main .wn-notification-message-container span{color:{$highlight_color} !important;}" . $custom_css_setting;

        $custom_css = str_replace("\n", '', $custom_css);
        $custom_css = str_replace("\r", '', $custom_css);
        $custom_css = str_replace("\t", '', $custom_css);
        $custom_css = str_replace("\l", '', $custom_css);
        $custom_css = str_replace("\0", '', $custom_css);
        wp_add_inline_style('woocommerce-notification', wp_kses_post($custom_css));
    }

    public function get_name_by_country( $country ) {
        if (!$country) {
            return false;
        }

        $name_by_country = $this->settings->get_params('name_by_country');
        if (empty($name_by_country) || !is_array($name_by_country)) {
            return false;
        }

        $names = '';
        foreach ($name_by_country as $rule) {
            if (empty($rule['countries'])) {
                continue;
            }

            if (in_array($country, $rule['countries']) && !empty($rule['names'])) {
                $names = $rule['names'];
                break;
            }
        }

        if (!empty($names)) {
            $names = explode("\n", $names);
            $names = array_filter($names);
            shuffle($names);
            $names = array_slice($names, 0, 50);
            $names = array_map('base64_encode', $names);
        }

        return $names;
    }

    public function get_start_virtual_time() {
        $current_hour = (int)gmdate('H', current_time('timestamp'));
        if ($current_hour >= 0 && $current_hour < 9) {
            $start = $current_hour + 2;
        } else {
            $start = 0;
        }

        return $start;
    }

    public function get_end_virtual_time() {
        $time = (int)$this->settings->get_params('virtual_time');
        $current_hour = (int)gmdate('H', current_time('timestamp'));
        if ($current_hour >= 0 && $current_hour < 9) {
            $end = $current_hour > $time ? $current_hour + $time + 2 : $time + 2;
        } else {
            if ($current_hour > $time) {
                $end = ($current_hour - $time) > 7 ? $time : $current_hour - 7;
            } else {
                $end = $time;
            }
        }

        return $end;
    }

}